#!/bin/bash

[[ "$TRACE" ]] && set -x

function usage() {
    echo "Usage: ./apply.sh [--check] [--fail-fast] [--filter=regexp]"
    echo "    --check - do not upload alerts to yasm, only render them. Enables --fail-fast"
    echo "    --fail-fast - fail on first error, do not process other files"
    echo "    --filter=regex - apply only files mathing regexp"
}

function http_call() {
    curl --silent --retry 2 --retry-delay 5 --fail --show-error "$@"
}

function http_call_with_response() {
    >/tmp/curl
    code=$(curl --write-out %{http_code} --silent --output /tmp/curl $@)
    if [[ $code =~ 2[0-9][0-9] ]]; then
        return 0;
    fi
    cat /tmp/curl
    return 1;
}

CHECK="false"
FAIL_FAST="false"
FILTER=".*"
SUCCESSFULL="true"

for i in "$@"
do
case $i in
    --check)
    CHECK="true"
    FAIL_FAST="true"
    shift
    ;;
    --fail-fast)
    FAIL_FAST="true"
    shift
    ;;
    --filter=*)
    FILTER="${i#*=}"
    shift
    ;;
    *)
        usage
        exit -1
    ;;
esac
done

function has_check_flag() {
    [[ $CHECK == "true" ]]
}

function has_fail_fast_flag() {
    [[ $FAIL_FAST == "true" ]]
}

function need_fail_exit() {
    [[ $SUCCESSFULL != "true" ]]
}

function render_alert() {
    alert_path=$1
    alert_name=$(echo $alert_path | sed 's|/|_|' )
    echo "rendering alert $alert_name"
    data=$(http_call_with_response --data-binary @$alert_path.j2 "https://yasm.yandex-team.ru/srvambry/tmpl/alerts/render_json/$alert_name")
    if [[ $? -ne 0 ]]; then
        echo "$alert_name broken, server response: $data"
        SUCCESSFULL="false"
        if has_fail_fast_flag; then exit -1; fi
    fi
    echo -n
}

function render_panel() {
    panel_path=$1
    panel_name=$(echo $panel_path | sed 's|/|_|' )
    echo "render panel $panel_name"
    data=$(http_call_with_response --data-binary @$panel_path.j2 "https://yasm.yandex-team.ru/srvambry/tmpl/panels/render_json/$panel_name")
    if [[ $? -ne 0 ]]; then
        echo "$panel_name broken, server response: $data"
        SUCCESSFULL="false"
        if has_fail_fast_flag; then exit -1; fi
    fi
    echo -n
}

function update_alert() {
    alert_path=$1
    alert_name=$(echo $alert_path | sed 's|/|_|' )

    echo "update alert $alert_name"
    data=$(http_call --data-binary @$alert_path.j2 "https://yasm.yandex-team.ru/srvambry/tmpl/alerts/update/content?key=$alert_name")
    if [[ $? != 0 ]]; then
        echo "update failed, server response: $data"
        SUCCESSFULL="false"
        if has_fail_fast_flag; then exit -1; fi
    fi
    echo -n
}

function apply_alert() {
    alert_path=$1
    alert_name=$(echo $alert_path | sed 's|/|_|' )
    echo "apply alert $alert_name"
    data=$(http_call -X POST "https://yasm.yandex-team.ru/srvambry/tmpl/alerts/apply/$alert_name")
    if [[ $? != 0 ]]; then
        echo "apply failed, server response: $data"
        SUCCESSFULL="false"
        if has_fail_fast_flag; then exit -1; fi
    fi
    echo -n
}

function update_panel() {
    panel_path=$1
    panel_name=$(echo $panel_path | sed 's|/|_|' )
    echo "update panel $panel_name"
    data=$(http_call --data-binary @$panel_path.j2 "https://yasm.yandex-team.ru/srvambry/tmpl/panels/update/content?key=$panel_name")
    if [[ $? != 0 ]]; then
        echo "update failed, server response: $data"
        SUCCESSFULL="false"
        if has_fail_fast_flag; then exit -1; fi
    fi
}

generators=($(ls generate* | grep "$FILTER"))
if [[ $? == 0 ]]; then
    for script in ${generators[@]}; do
        ./${script}
        if [[ $? != 0 ]]; then
            echo "generate failed for ${script}"
            SUCCESSFULL="false"
            if has_fail_fast_flag; then exit -1; fi
        fi
    done
fi

#these commons for alerts and panel so use update_panel and update_alert

if ! has_check_flag; then
    commons=($(ls sre_common_vars.j2 *common.j2 | grep "$FILTER"))
    if [[ $? == 0 ]]; then
        for file in ${commons[@]}; do
            update_panel ${file%.*}
            update_alert ${file%.*}
        done
    fi
fi

panels=$(ls *panel.j2 marty/*panel.j2| grep "$FILTER")
if [[ $? == 0 ]]; then
    for panel in ${panels[@]}; do
        if has_check_flag; then
            render_panel ${panel%.*}
        else
            update_panel ${panel%.*}
        fi
    done
fi


if ! has_check_flag; then
    alerts=$(ls *alerts.j2 marty/*alerts.j2 | grep "$FILTER")
    if [[ $? == 0 ]]; then
        for alert in ${alerts[@]}; do
            update_alert ${alert%.*}
        done
    fi
fi

alerts=$(ls *alerts.j2 marty/*alerts.j2 | grep "$FILTER" | sed '/sre_common_alerts.j2/d' | sed '/sre_nginx_[45]xx_alerts.j2/d')
if [[ $? == 0 ]]; then
    for alert in ${alerts[@]}; do
        if has_check_flag; then
            render_alert ${alert%.*}
        else
            apply_alert ${alert%.*}
        fi
    done
fi

if need_fail_exit; then
    exit -1;
fi
