#! /usr/bin/env python3

import os
import sys
import requests
import json
import re


class Cluster:
    def __init__(self, id, name, databases=[], responsible=[]):
        self.Id = id
        self.Name = name
        self.Databases = databases
        self.Responsible = responsible


class ClusterGroup:
    def __init__(self, cluster_names=[]):
        self.ClusterNames = cluster_names


class Folder:
    def __init__(self, id, name, clusters=[], groups={}):
        self.Id = id
        self.Name = name
        self.Clusters = clusters
        self.Groups = groups


class DatabaseMapping:
    def __init__(self, folders=[]):
        self.Folders = folders

    def dump(self, file=sys.stdout):
        print(
            '<% set mapping = {mapping} %>'.format(
                mapping=json.dumps(
                    self,
                    indent=4,
                    cls=MappingEncoder,
                    ensure_ascii=False
                )
            ),
            file=file,
        )


class MappingEncoder(json.JSONEncoder):
    def default(self, obj):
        if isinstance(obj, DatabaseMapping):
            return obj.Folders
        elif isinstance(obj, Folder):
            return dict(name=obj.Name, id=obj.Id, clusters=obj.Clusters, groups=obj.Groups)
        elif isinstance(obj, Cluster):
            return dict(name=obj.Name, id=obj.Id, databases=obj.Databases, responsible=obj.Responsible)
        elif isinstance(obj, ClusterGroup):
            return list(obj.ClusterNames)

        return json.JSONEncoder.default(self, obj)


class ClusterInfoReader():
    def __init__(self, oauthToken):
        self.IamTokenUrl = 'https://iam.cloud.yandex-team.ru/v1/tokens'
        self.ClustersListUrl = 'https://gw.db.yandex-team.ru/managed-postgresql/v1/clusters'
        self.DatabasesListUrl = 'https://gw.db.yandex-team.ru/managed-postgresql/v1/clusters/{clusterId}/databases'
        self.IamToken = None
        self.OauthToken = oauthToken

    def get_mapping(self, config):
        return DatabaseMapping(folders=[
            self._get_folder(folder['name'], folder['id'], folder['clusters'], folder['groups'])
            for folder in config['folders']
        ])

    def _get_folder(self, name, id, filter, groups):
        clusters = self._get_clusters(folderId=id, filter=filter)
        return Folder(
            id=id,
            name=name,
            clusters=clusters,
            groups=self._get_groups(groups=groups, clusters=clusters))

    def _get_clusters(self, folderId, filter):
        url = self.ClustersListUrl
        params = {'folder_id': folderId}
        headers = {'Authorization': 'Bearer {token}'.format(token=self._iam_token())}
        resp = requests.get(url, params=params, headers=headers, timeout=60, verify=False)
        resp.raise_for_status()

        respJson = json.loads(resp.text)
        assert 'nextPageToken' not in respJson

        clusters = []
        for cluster in respJson['clusters']:
            clusterInfo = next((x for x in filter if re.search(x['name_regex'], cluster['name'])), None)
            if not clusterInfo:
                continue

            id = cluster['id']
            name = cluster['name']
            databases = self._get_databases(clusterId=id)
            responsible = clusterInfo['responsible']
            if isinstance(responsible, str):
                responsible = re.split(r'\s*,\s*', responsible)

            clusters.append(Cluster(id=id, name=name, databases=databases, responsible=responsible))

        return clusters

    def _get_groups(self, groups, clusters):
        res = {}
        for name, regex in groups.items():
            cluster_names = [c.Name for c in clusters if re.match(regex, c.Name)]
            if cluster_names:
                res[name] = ClusterGroup(cluster_names=cluster_names)
        return res

    def _get_databases(self, clusterId):
        url = self.DatabasesListUrl.format(clusterId=clusterId)
        headers = {'Authorization': 'Bearer {token}'.format(token=self._iam_token())}
        resp = requests.get(url, headers=headers, timeout=60, verify=False)
        resp.raise_for_status()

        respJson = json.loads(resp.text)
        assert 'nextPageToken' not in respJson

        return [database['name'] for database in respJson['databases']]

    def _iam_token(self):
        if not self.IamToken:
            self.IamToken = self._get_iam_token(self._oauth_token())

        return self.IamToken

    def _get_iam_token(self, oauthToken):
        url = self.IamTokenUrl
        headers = {'Content-Type': 'application/json'}
        data = json.dumps({'yandexPassportOauthToken': oauthToken})
        resp = requests.post(url, headers=headers, data=data, timeout=30, verify=False)
        resp.raise_for_status()

        return json.loads(resp.text)['iamToken']

    def _oauth_token(self):
        return self.OauthToken


config = {
    "folders": [
        {
            "name": "mail",
            "id": "foom5upuus069lavolqb",
            "clusters": [
                {
                    "name_regex": "^collie_db$",
                    "responsible": "@svc_collie_development"
                },
                {
                    "name_regex": "^mopsdb-(intranet-)?production$",
                    "responsible": "@svc_mops_development"
                },
                {
                    "name_regex": "^rpopdb_transfer[0-9]{2}$",
                    "responsible": "@svc_collectors_development"
                },
                {
                    "name_regex": "^act_db$",
                    "responsible": "vyserenity"
                },
                {
                    "name_regex": "^botdb-(corp|production)$",
                    "responsible": "grascm"
                }
            ],
            "groups": {
                "rpopdb_transfer": "^rpopdb_transfer[0-9]{2}$"
            }
        },
        {
            "name": "xiva",
            "id": "foocn84l3bfvv91tsdkb",
            "clusters": [
                {
                    "name_regex": "^xiva_((xstore|xtable)_(corp|production_[0-9]{2})|conf|resharddb_production)$",
                    "responsible": "@svc_xiva_development"
                }
            ],
            "groups": {}
        },
        {
            "name": "collectors",
            "id": "foosaanhpqhofon45s8d",
            "clusters": [
                {
                    "name_regex": "^rpopdb_production_[0-9]{2}$",
                    "responsible": "@svc_collectors_development"
                }
            ],
            "groups": {
                "rpopdb_production": "^rpopdb_production_[0-9]{2}$"
            }
        }
    ]
}


if __name__ == '__main__':
    oauthToken = os.environ.get('QLOUD_ROBOT_MAIL_SRE_OAUTH_TOKEN')
    outputFilePath = 'database_common.j2'

    clusterReader = ClusterInfoReader(oauthToken=oauthToken)
    mapping = clusterReader.get_mapping(config)

    outputFile = open(outputFilePath, 'w') if outputFilePath else None
    mapping.dump(file=outputFile)
