#! /usr/bin/env python3

import os
import requests
import json
import re

QLOUD_API_URL = "https://qloud-ext.yandex-team.ru/"

qloud_services = {
    ('production', 'corp'): [
        'mbody',
        'imap',
        'xiva-server',
        'xivahub',
        'xivamob',
        'mailpusher',
        'equalizer'
    ],
    ('production', 'intranet-production'): [
        'akita',
        'hound',
        'sendbernar',
        'mops',
        'settings',
        'u2709',
        'quinn',
        'web-api',
        'mobile-api',
        'webmail-api-v1',
        'homer',
        'catdog'
    ],
    ('production', 'canary'): [
        'mdbsave',
        'mdbsave-corp',
        'notsolitesrv',
        'notsolitesrv-corp',
        'nwsmtp',
        'nwsmtp-corp'
    ],
    ('production', ): [
        'pop3',
        'collectors',
        'xeno',
        'beagle',
        'gendarme',
        'ipa',
        'setter',
        'maillists',
        'fouras',
        'calendar-public',
        'calendar-yt',
        'maya',
        'maya-corp',
        'reminders',
        'ratesrv',
        'furita',
        'ratesrv-corp',
        'furita-corp',
    ],
    ('production', 'import'):
        ['collectors-ext']
}

DC_DISTRIBUTION_DICT = {}


class MappingEncoder(json.JSONEncoder):
    def default(self, obj):
        if isinstance(obj, set):
            return list(obj)
        return json.JSONEncoder.default(self, obj)


qloud_enviroments_ids = []
for service in qloud_services:
    for env in service:
        for component in qloud_services[service]:
            qloud_enviroments_ids.append(f"mail.{component}.{env}")


def get_data_by_enviroment(env: str) -> requests.request:
    try:
        return requests.get(QLOUD_API_URL + "api/v1/environment/stable/" + env,
                            headers=QLOUD_AUTH_HEADERS, timeout=(3, 10))
    except (requests.exceptions.ConnectTimeout, requests.exceptions.ReadTimeout):
        print('Qloud timeout, use old data')
        exit(0)  # finish script successfully and use old data


def save_mapping(file, data: dict) -> None:
        print(
            '<% set mapping = {mapping} %>'.format(
                mapping=json.dumps(
                    data,
                    indent=4,
                    cls=MappingEncoder,
                    ensure_ascii=False
                )
            ),
            file=file,
        )


def count_dcs_by_component(component: dict) -> set:
    return {
        re.search(r"^(iva|myt|sas|vla|man)?", instance.get("host", "")).group()
        for instance in component['runningInstances']
    } - {''}

component_mapping = [
        (
            r'^mail\.(?P<app>nwsmtp(?:-corp)?)\.(?P<env>production|canary)\.(?P<comp>.*)',
            r'mail.\g<app>.production.\g<comp>'
        ),
        (
            r'^mail\.(?P<app>(?:mdbsave|notsolitesrv)(?:-corp)?)\.(?P<env>production|canary)\.(?P<comp>.*)',
            r'mail.\g<app>.production.\g<comp>'
        ),
        (
            r'^mail\.(?P<app>[^.]+)\.(?P<env>[^.]+)\.(?P<comp>.*)',
            r'mail.\g<app>.\g<env>'
        )
    ]


if __name__ == '__main__':
    QLOUD_TOKEN = os.environ.get('QLOUD_ROBOT_MAIL_SRE_OAUTH_TOKEN')
    QLOUD_AUTH_HEADERS = {"Authorization": "OAuth " + QLOUD_TOKEN}
    outputFilePath = 'mail_status_panel_common.j2'

    for env in qloud_enviroments_ids:
        response = get_data_by_enviroment(env)
        if response.status_code == 200:
            components_list = response.json()["components"]
            for component in components_list:
                for p, s in component_mapping:
                    component_id = f"{env}.{component}"
                    if re.match(p, component_id):
                        name = re.sub(p, s, component_id)
                        dc_distribution = count_dcs_by_component(components_list[component])
                        if dc_distribution:
                            DC_DISTRIBUTION_DICT.setdefault(name, set()).update(dc_distribution)
                        break

    outputFile = open(outputFilePath, 'w')
    save_mapping(outputFile, DC_DISTRIBUTION_DICT)

