from datetime import datetime
from typing import List

import attr
import cattr

from http_clients.base import HttpClient
from http_clients.calendar import Person
from utils.config import config
from utils.utils import datetime_converter


@attr.s(auto_attribs=True, frozen=True)
class Schedule:
    id: int
    name: str


@attr.s(frozen=True)
class Shift:
    id: int = attr.ib()
    is_approved: bool = attr.ib()
    start: str = attr.ib()
    end: str = attr.ib()
    start_datetime: datetime = attr.ib(converter=datetime_converter)
    end_datetime: datetime = attr.ib(converter=datetime_converter)

    schedule: Schedule = attr.ib()
    person: Person = attr.ib()
    replaces: List["Shift"] = attr.ib(default=attr.Factory(list))


class AbcClient(HttpClient):
    base_url = config.abc.url or "https://abc-back.yandex-team.ru/api/v4"
    decode_json = True
    oauth = config.abc_token
    date_format = "%Y-%m-%d"

    def duty_shifts(self, service_slug: str, date_from: datetime, date_to: datetime, **kwargs) -> List[Shift]:
        response = self.r.get(
            "/duty/shifts/",
            params={
                "service__slug": service_slug,
                "date_from": date_from.strftime(self.date_format),
                "date_to": date_to.strftime(self.date_format),
                **kwargs,
            },
        )

        # резолвим аннотацию List["Shift"] в тип
        attr.resolve_types(Shift, globals(), locals())
        return cattr.structure(response.get("results"), List[Shift])
