from functools import wraps
from itertools import repeat
from typing import Any, Optional, Tuple, Union
from urllib.parse import urljoin

import requests
from loguru import logger
from requests import Response
from requests.adapters import HTTPAdapter
from urllib3 import Retry

from utils.config import config
from utils.tvm import get_tvm_headers


class HttpClientSession(requests.Session):
    def __init__(
        self,
        /,
        timeout: Optional[Union[Tuple[Union[int, float], Union[int, float]], int, float]] = None,
        base_url: Optional[str] = None,
    ):
        self.timeout = timeout
        self.base_url = base_url
        super().__init__()

    def request(self, method, url, *args, **kwargs):
        """
        Send the request after generating the complete URL.
        """
        url = self.create_url(url)
        kwargs.setdefault("timeout", self.timeout)
        return super().request(method, url, *args, **kwargs)

    def create_url(self, url):
        """
        Create the URL based off this partial path.
        """
        return urljoin(self.base_url.rstrip('/') + '/', url.lstrip('/')) if self.base_url else url


class HttpClient:
    """
    Base class for HTTP(s) clients

    Supported options and example:

    class SimpleClient(HttpClient):
        base_url = "https://ya.ru/" # BaseURL string (path will be normalized with url)
        oauth = "xxxxADS_Sdsdasdsdsdasdasda" # Additional Authorization header
        headers = {"Session_id": "example"}  # Any additional headers dict
        retries = 0 # Disable retries or integer for retry value (default=3)
        timeout = 1 # Timeout in seconds or tuple (read, connect) timeouts (default=4)
        retry_if_status = (400) # Tuple of http status codes default: (500, 502, 504)
        decode_json = True # Auto-magically convert returned json
        retry_backoff_factor = 0.1 # https://nda.ya.ru/t/AI8IVeio3YjEHC
    """

    base_url: Optional[str] = None
    oauth: Optional[str] = None
    timeout: Optional[Union[Tuple[Union[int, float], Union[int, float]], int, float]] = 3
    retries: int = 3
    retry_if_status: Tuple[int, ...] = (500, 502, 504)
    retry_backoff_factor: float = 0.3
    decode_json: bool = False

    def __init__(self):
        self.log = logger
        self.r = HttpClientSession(timeout=self.timeout, base_url=self.base_url)

        self.r.headers.update({"X-Host": config.hostname, "User-Agent": config.useragent})
        self.r.headers.update(getattr(self, "headers", {}))

        if self.oauth:
            self.r.headers.update({"Authorization": f"OAuth {self.oauth}"})

        if self.retries:
            self.mount_retries()

        self.decorate_session()

    def mount_retries(self):
        r = Retry(
            total=self.retries,
            read=self.retries,
            connect=self.retries,
            backoff_factor=self.retry_backoff_factor,
            status_forcelist=self.retry_if_status,
            method_whitelist=False,
        )
        adapter = HTTPAdapter(max_retries=r)
        map(self.r.mount, *zip(("http://", "https://"), repeat(adapter)))

    def decorate_session(self):
        def request_decorator(method):
            @wraps(method)
            def wrapper(*args, tvm_client_id: Optional[str] = None, tvm_robot_user_ticket: bool = False, **kwargs) -> \
            Union[Response, Any]:
                kwargs.setdefault("headers", {})
                # можно было бы использовать алиасы, но tvm2 либа поддерживает только id, иначе пришлось бы свой tvm клиент писать
                if tvm_client_id:
                    kwargs["headers"].update(get_tvm_headers(tvm_client_id, tvm_robot_user_ticket))

                try:
                    r = method(*args, **kwargs)
                    self.log.debug(
                        f'{args[0]} request completed with status: {getattr(r, "status_code", "")} with base: {self.base_url} to: {args[1]}'
                    )
                except Exception as e:
                    self.log.error(f"{args[0]} request failed with base: {self.base_url} to: {args[1]}")
                    self.log.exception(e)
                    raise e

                if getattr(self, "decode_json", None):
                    try:
                        self.log.debug(f"decoding json from response with {r.status_code} status code")
                        r = r.json()
                    except Exception as e:
                        self.log.exception(e)
                        raise e
                return r

            return wrapper

        self.r.request = request_decorator(self.r.request)

    # Cleanup session on object deletion
    def __del__(self):
        self.r.close()
        s = super()
        if hasattr(s, "__del__"):
            s.__del__(self)
