from datetime import datetime
from typing import List, Dict, Any, Set, Union

import attr

from http_clients.base import HttpClient
from utils.config import config
from utils.utils import converter, datetime_converter


@attr.s(auto_attribs=True, eq=False, frozen=True, str=False, repr=False)
class Person:
    login: str
    uid: int
    name: str = ""
    email: str = ""
    decision: str = ""
    office_id: str = ""

    def __eq__(self, other: Any) -> bool:
        return self.uid == other.uid

    __contains__ = __eq__

    def __str__(self):
        return f"{self.login}@"

    __repr__ = __str__


@attr.s(str=False, repr=False)
class Event:
    id: int = attr.ib()
    external_id: str = attr.ib()
    sequence: int = attr.ib()
    name: str = attr.ib()
    type: str = attr.ib()
    description: str = attr.ib()
    location: str = attr.ib()
    layerId: int = attr.ib()

    start_ts: datetime = attr.ib(converter=datetime_converter)
    end_ts: datetime = attr.ib(converter=datetime_converter)
    instance_start_ts: datetime = attr.ib(converter=datetime_converter)

    notifications: List[Dict] = attr.ib()
    actions: Dict = attr.ib()
    resources: List = attr.ib()
    data: Dict = attr.ib()

    total_attendees: int = attr.ib()
    total_optional_attendees: int = attr.ib()

    organizer: Person = attr.ib()
    attendees: List[Person] = attr.ib(default=attr.Factory(list))
    optional_attendees: List[Person] = attr.ib(default=attr.Factory(list))
    subscribers: List[Person] = attr.ib(default=attr.Factory(list))
    repetition: Dict = attr.ib(default=attr.Factory(dict))

    def __str__(self):
        return f"Event id:{self.id} \"{self.name}\" {self.start_ts.strftime('%Y-%m-%d %H:%M')}"

    __repr__ = __str__

    def get_updated_attendees(self, new_attendees: Set) -> Set[str]:
        new_attendees.update({f"{a.login}@yandex-team.ru" for a in self.attendees})
        # Чтобы указать переговорки, необходимо указать их почтовые адреса в поле attendees.
        # если не добавить, то переговорка удалится из встречи
        rooms = {resource["email"] for resource in self.resources if resource["resourceType"] == "room"}
        new_attendees.update(rooms)
        return new_attendees


class CalendarClient(HttpClient):
    base_url = config.calendar.url
    tvm_client_id = config.calendar.tvm_client_id
    decode_json = True
    uid = config.calendar.robot_mail_sre_uid

    def update_event(
        self,
        event_id: int,
        attendees: Union[List[str], Set[str]],
        start_ts: datetime,
        end_ts: datetime,
        instance_start_ts: datetime,
        description: str = None,
        **kwargs,
    ) -> Dict:

        data = {
            "startTs": start_ts.isoformat(),
            "endTs": end_ts.isoformat(),
            "attendees": list(attendees),
            "description": description,
            **kwargs,
        }
        response = self.r.get(
            "/update-event",
            params={
                "uid": self.uid,
                "id": event_id,
                "instanceStartTs": instance_start_ts.isoformat(),
                "tz": "Europe/Moscow",
            },
            json=data,
            tvm_client_id=self.tvm_client_id,
            tvm_robot_user_ticket=True,
        )

        return response

    def get_event(self, event_id: int, start_ts: str = None) -> Event:
        response = self.r.get(
            "/get-event",
            params={
                "uid": self.uid,
                "eventId": event_id,
                "instanceStartTs": start_ts,
            },
            tvm_client_id=self.tvm_client_id,
            tvm_robot_user_ticket=True,
        )

        return converter.structure(response, Event)

    def get_events(self, begin_time: datetime, end_time: datetime, layer_id: int = None, event_id: int = None):
        response = self.r.get(
            "/get-events",
            params={
                "uid": self.uid,
                "layerId": layer_id,
                "from": begin_time.isoformat(timespec="seconds"),
                "to": end_time.isoformat(timespec="seconds"),
                "tz": "Europe/Moscow",
                "eventId": event_id,
            },
            tvm_client_id=self.tvm_client_id,
            tvm_robot_user_ticket=True,
        )

        return converter.structure(response.get("events", []), List[Event])
