from datetime import timedelta, datetime
from enum import Enum
from typing import List, Dict

import attr

from http_clients.base import HttpClient
from utils.config import config
from utils.utils import WardenPeriod, converter


class WardenCheckType(str, Enum):
    ALERT = 'ALERT'
    FUNCTIONALITIES = 'FUNCTIONALITIES'


class WardenCheckStatus(str, Enum):
    OK = 'OK'
    ERROR = 'ERROR'


# frozen=True для хеширования и использования как ключ в словаре
# если что можно поменять на hash=False, и определить метод __hash__
@attr.s(frozen=True, str=False, repr=False)
class Issue:
    key: str = attr.ib()
    status: str = attr.ib()
    updated: int = attr.ib()
    created: int = attr.ib()
    summary: str = attr.ib()

    def __str__(self):
        return f"{self.__class__.__name__}: {self.key}"

    __repr__ = __str__


@attr.s(str=False, repr=False)
class ActionItem(Issue):
    priority: str = attr.ib()
    lsr_action_item: bool = attr.ib(default=False)
    resolved: bool = attr.ib(default=False)


@attr.s(str=False, repr=False)
class SPProblem(Issue):
    resolved: bool = attr.ib(default=False)


@attr.s(str=False, repr=False)
class Incident(Issue):
    markup: Dict = attr.ib()
    priority: str = attr.ib()

    action_items: List[ActionItem] = attr.ib(default=attr.Factory(list))
    problems: List[SPProblem] = attr.ib(default=attr.Factory(list))
    stats: Dict = attr.ib(default=attr.Factory(dict))

    author: str = attr.ib(default="")
    duty: str = attr.ib(default="")
    assignee: str = attr.ib(default="")

    yandex_downtime: float = attr.ib(default=0.0)
    resolution: str = attr.ib(default="")
    solved: bool = attr.ib(default=False)
    is_lsr_candidate: bool = attr.ib(default=False)
    solved_time: int = attr.ib(default=0)
    incident_time: int = attr.ib(default=0)


@attr.s(str=False, repr=False)
class ComponentCheckCauses:
    alert_ids: List[str] = attr.ib(default=attr.Factory(list))

    def __str__(self):
        return ' '.join(self.alert_ids)

    __repr__ = __str__


@attr.s(str=False, repr=False)
class ComponentCheck:
    status: str = attr.ib()
    description: str = attr.ib()
    checkType: str = attr.ib(default='')
    causes: ComponentCheckCauses = attr.ib(default=ComponentCheckCauses)

    def __str__(self):
        return f"{self.checkType} {self.status} {self.causes}"

    __repr__ = __str__


@attr.s(eq=False)
class ComponentTag:
    tag: str = attr.ib(default='')

    def __eq__(self, other):
        return self.tag == other


@attr.s()
class ComponentDocumentation:
    objects: List = attr.ib(default=[])


@attr.s(frozen=True, str=False, repr=False)
class Component:
    name: str = attr.ib()
    abcServiceSlug: str = attr.ib(default='')
    component_checks: List[ComponentCheck] = attr.ib(default=attr.Factory(list))
    tags: List[ComponentTag] = attr.ib(default=attr.Factory(list))
    documentation: ComponentDocumentation = attr.ib(default=ComponentDocumentation())


class WardenClient(HttpClient):
    base_url = config.warden.url or "https://warden.z.yandex-team.ru/api/warden.Warden"
    decode_json = True
    timeout = 15.0
    oauth = config.warden_token

    def get_incident_list(
        self, component_name: str, parent_component_name: str, period: str = "", created_time: dict = None
    ) -> List[Incident]:
        if not period:
            period = WardenPeriod.review

        # в warden чтобы запросить какой-то кастомный период надо указать "period": "custom" а
        # а в поле "created" отправить нужные даты в timestamp
        if period == WardenPeriod.custom and not created_time:
            now = datetime.now()
            created_time = {
                "start": int((now - timedelta(days=7)).timestamp()),
                "end": int(now.timestamp()),
            }

        url = "/getIncidentList"
        json = {
            "componentName": component_name,
            "parentComponentName": parent_component_name,
            "period": period.upper(),
            "created": created_time,
        }

        response = self.r.post(url=url, json=json)
        return converter.structure(response.get("incidents", []), List[Incident])

    def get_component(self, component_name: str = None, parent_component_name: str = None, slug: str = None) -> Component or None:
        if not component_name and not parent_component_name and not slug:
            return None

        url = "/getComponent"
        json = {
            "componentName": component_name,
            "parentComponentName": parent_component_name,
            "slug": slug
        }

        response = self.r.post(url=url, json=json)

        return converter.structure(response.get("component", {}), Component)

    def get_child_components(self, parent_component_name: str = None, slug: str = None) -> List[Component]:
        if not parent_component_name and not slug:
            return []

        url = "/getChildComponents"
        json = {
            "parentComponentName": parent_component_name,
            "parentSlug": slug
        }

        response = self.r.post(url=url, json=json)
        return converter.structure(response.get("objects", []), List[Component])

