import re
from abc import abstractmethod
from datetime import datetime
from typing import Dict, Optional

from loguru import logger

from http_clients.locke import yt
from utils.config import Config, ConfigEntry


class BaseJob:
    def __init__(self, config: Config):
        self._name = re.sub("Job$", "", self.__class__.__name__).lower()   # считаем что все задачи будут называться <Name>Job
        self._locke_path = f"{config.yt.path}/{self._name}"  # путь к локам для данной задачи
        self.config = config.get(self._name) # настройки джобы
        self.scheduler_settings = self.get_scheduler_settings(self.config)
        self.yt = yt
        self.acquire_lock_path = f"{self._locke_path}/lock"
        self.hostname = config.hostname
        self.default_node_type = "map_node"
        self.string_node_type = "string_node"

    @abstractmethod
    def run_once(self):
        pass

    def run(self):
        if self.config.get("disabled"):
            logger.info(f"{self._name} job is disabled in config")
            return

        logger.info(f"{self._name} job is starting")
        if not self.acquire_lock():
            logger.info("can not acquire lock, lock is exist")
            return
        try:
            self.run_once()
        except Exception as e:
            logger.exception(f"job execution error: {e}")
        finally:
            self.remove_lock()
            logger.info(f"{self._name} job is finished")

    def get_scheduler_settings(self, config: Optional[ConfigEntry]) -> Dict:
        settings = {
            "hours": 1,
            "trigger": "interval",
            "id": self._name,
            "next_run_time": datetime.now(),  # https://stackoverflow.com/questions/43254531
        }

        if config.scheduler.trigger:
            # удаляем дефолтное значение scheduler-a, если описан в конфиге
            settings.pop("hours")

        settings.update(config.scheduler.as_dict())

        if config.scheduler.not_start_now:
            # не запускаем сразу, а только по расписанию
            settings.pop("next_run_time")
            settings.pop("not_start_now")

        return settings

    def acquire_lock(self) -> bool:
        if not self.yt.exists(self.acquire_lock_path):
            self.yt.create(
                type=self.default_node_type,
                path=self.acquire_lock_path,
                recursive=True,
                attributes={"host": self.hostname},
            )
            return True
        return False

    def remove_lock(self) -> None:
        self.yt.remove(path=self.acquire_lock_path, force=True)

    def create_node(self, path: str, node_name: str, recursive: bool = True):
        full_path = f"{path}/{node_name}"
        id = self.yt.create(type=self.default_node_type, path=full_path, recursive=recursive)
        logger.debug(f"object created: {id}, path: {full_path}")
