from typing import Dict, List, Set

from loguru import logger
from startrek_client import Startrek

from http_clients.warden import WardenClient, Incident, ActionItem, SPProblem
from jobs import BaseJob
from utils.config import Config
from utils.templates import render_template


class FindSPProblemsJob(BaseJob):
    def __init__(self, config: Config):
        super().__init__(config)
        self.startrek = Startrek(useragent=config.useragent, token=config.st_token)
        self.warden = WardenClient()
        self.components = config.components
        self.spi_count_for_problem = 2  # количество инцидентов с одним и тем же AI, при котором пора выделять SPProblem

    def run_once(self):
        for component in self.components:
            logger.info(f"start processing component: {component}")

            # у нас только рутовые компоненты
            incidents = self.warden.get_incident_list(component_name=component, parent_component_name="")

            all_action_items = self.get_sorted_action_items(incidents)

            for action_item, linked_incidents in all_action_items.items():
                if action_item.status == "Закрыт":
                    continue
                self.problem_identification(action_item, linked_incidents)

            logger.info(f"finished processing component: {component}")

    def get_uniq_linked_problems(self, incidents: List[Incident]) -> Set[SPProblem]:
        # в теории может быть у одного инцидента несколько разных проблем, но на практике такого не видел
        return {
            problem
            for incident in incidents
            for problem in incident.problems
        }

    def get_sorted_action_items(self, incidents: List[Incident]) -> Dict[ActionItem, List[Incident]]:
        all_action_items: Dict[ActionItem, List[Incident]] = {}

        for incident in incidents:
            for ai in incident.action_items:
                all_action_items.setdefault(ai, []).append(incident)

        return all_action_items

    def problem_identification(self, action_item: ActionItem, linked_incidents: List[Incident]):
        current_incident_count = len(linked_incidents)
        exist_problems = self.get_uniq_linked_problems(linked_incidents)

        logger.info(f"{action_item} has {current_incident_count} linked incidents")
        logger.info(f"there are {len(exist_problems)} problems in {action_item} related incidents")

        # если spproblem уже прилинкована к каждому инциденту (кол-во меньше чем кол-во инцидентов)
        # то не выделяем новую, при этом считаем, что к одному инциденту прилинкована обычно одна проблема
        if len(exist_problems) < current_incident_count >= self.spi_count_for_problem:
            logger.info(f"{action_item} can turn into a SPProblem")

            ai_locke_path = f"{self._locke_path}/{action_item.key}"
            if not self.yt.exists(ai_locke_path):
                # ставим комментарий в крайний инцидент и записываем инциденты в YT как обработанные
                self.notify_about_new_problem(action_item, linked_incidents, ai_locke_path)
                return

            # проверяем нет ли новых инцидентов с существующим AI
            self.checking_for_new_incidents(action_item, linked_incidents, ai_locke_path)

    def checking_for_new_incidents(self, action_item: ActionItem, linked_incidents: List[Incident], ai_locke_path: str):
        for incident in linked_incidents:
            if not self.yt.exists(f"{ai_locke_path}/{incident.key}"):
                logger.info(f"send comment to {incident.key}")
                self.create_node(ai_locke_path, incident.key)
                self.send_comment_about_problem(incident.key, action_item.key, linked_incidents)
                continue

            logger.debug(f"incident {incident.key} was processed before")

    def notify_about_new_problem(self, action_item: ActionItem, linked_incidents: List[Incident], ai_locke_path: str):
        for incident in linked_incidents:
            self.create_node(ai_locke_path, incident.key)

        last_incident = linked_incidents[-1]
        logger.info(f"send comment to last incident: {last_incident}")
        self.send_comment_about_problem(last_incident.key, action_item.key, linked_incidents)

    def send_comment_about_problem(self, incident_issue: str, action_item_issue: str, incidents: List[Incident]):
        context = {
            "incidents": "\n".join([incident.key for incident in incidents]),
            "action_item": action_item_issue,
        }

        text = render_template(name="notify_about_problem.jinja2", context=context)

        logger.debug(f"creating a comment: {text} in the ticket: {incident_issue=}")

        issue = self.startrek.issues[incident_issue]
        comment = issue.comments.create(text=text)
        logger.info(f"comment is created: {comment}")
