import re
from datetime import datetime, timedelta
from typing import Dict

from loguru import logger
from startrek_client import Startrek

from http_clients.calendar import CalendarClient, Event
from http_clients.yaincbot import YaIncBotClient
from jobs import BaseJob
from utils.config import Config
from utils.templates import render_template


class MeetingsAnnouncementsJob(BaseJob):
    def __init__(self, config: Config):
        super().__init__(config)
        self.calendar = CalendarClient()
        self.bot = YaIncBotClient()
        self.startrek = Startrek(useragent=config.useragent, token=config.st_token)
        self.date_lock_format = "%Y-%m-%d"
        self.date_template_format = "%d.%m"

    def run_once(self):
        now = datetime.now()

        for event_config in self.config.events:
            # получаем ближайшую встречу из календаря
            event = self.calendar.get_event(event_id=event_config.get("id"))
            future_events_period = event_config.get("future_events_period", {"days": 1})

            if (now + timedelta(**future_events_period)) < event.start_ts:
                continue

            # ключ для того чтобы проверять отправили ли оповещение
            # вида {event_id}_{%Y-%m-%d"} - две встречи одинаковых в данном случае не может быть
            event_uniq_key = f"{event.id}_{event.start_ts.strftime(self.date_lock_format)}"
            if self.yt.exists(f"{self._locke_path}/{event_uniq_key}"):
                logger.info(f"for {event} notify has already been sent")
                continue

            context = self.building_template_context(event)
            if not context:
                continue

            msg = render_template(name="announcement.jinja2", context=context)
            chats = event_config.get('chats')
            if not chats:
                continue

            logger.debug(f"send message: {msg} to chats: {chats}")
            # создаем лок, что отправили сообщение, да бы избежать дублирования, если что-то пойдет не так
            self.create_node(path=self._locke_path, node_name=event_uniq_key)
            try:
                bot_response = self.bot.send_notify(msg=msg, chats=chats)
                logger.debug(f"response from YaIncBot: {bot_response}")
            except Exception as e:
                logger.error(f'unexpected error when sending: {e}')

    def get_issues_from_event(self, event: Event) -> Dict[str, str]:
        """получаем из описания встречи инциденты формируем словарь вида {'SPI-xxx': 'название SPI'}"""
        incidents = {}

        for incident_key in re.findall("SPI-\d+", event.description):
            issue = self.startrek.issues[incident_key]
            # убираем из названия дату инцидента
            summary = re.split("^\[\d{4}\.\d{2}\.\d{2}\]", issue.summary)[-1]
            incidents[issue.key] = summary.strip()

        return incidents

    def building_template_context(self, event: Event) -> Dict[str, str]:
        incidents = self.get_issues_from_event(event)
        if not incidents:
            return {}

        return {
            "event_id": event.id,
            "incidents": incidents,
            "event_name": event.name,
            "number_of_weekday": event.start_ts.weekday(),
            "date": event.start_ts.strftime(self.date_template_format),
        }
