from datetime import datetime
from typing import Optional
from http_clients.infra import InfraClient
from http_clients.uchenki import UchenkiClient
from jobs import BaseJob
from loguru import logger
from startrek_client import Startrek
from utils.config import Config
from utils.utils import DCNames, UchenkiStatus


class SendDrillsResultsJob(BaseJob):
    def __init__(self, config: Config):
        super().__init__(config)
        self.infra_client = InfraClient()
        self.client = Startrek(useragent=config.useragent, token=config.st_token)
        self.uchenki_client = UchenkiClient()
        self.components = config.components
        self.date_format = "%Y-%m-%d"

    def parse_infra(self, ticket) -> (Optional[str], Optional[str]):
        date = None

        for link in ticket.remotelinks:
            if 'yandex-infra' in link.object.self and link.object.status['name'] == 'Resolved':
                infra_data = self.infra_client.get_event(link.object.id)
                date = datetime.utcfromtimestamp(infra_data['start_time']).strftime(self.date_format)
                if 'проверка ручки деградации' in infra_data['title'].lower():
                    break
                if infra_data and ticket.key in infra_data['tickets']:
                    for dc in DCNames.list():
                        if infra_data[dc]:
                            return dc, date
        return None, date

    def get_incidents(self, drill_ticket, drill_ticket_components: list) -> dict:
        found_spi = {}

        for link in drill_ticket.links:
            link_url = link.object.self
            if 'SPI' in link_url:
                spi_key = link_url.split('/')[-1]
                spi_ticket = self.client.issues[spi_key]
                for component in spi_ticket.components:
                    if component.name in drill_ticket_components:
                        found_spi.setdefault(component.name, []).append(spi_key)
        return found_spi

    def set_tag(self, ticket_key: str) -> None:
        self.client.issues[ticket_key].update(
            tags={'add': [self.config.already_sent_tag]}
        )
        logger.info(f"set {self.config.already_sent_tag} tag to ticket {ticket_key}")

    def process_drill_ticket(self, drill_ticket):
        ticket_key = drill_ticket.key
        logger.info(f"start processing drill ticket: {ticket_key}")
        drill_ticket_components = [tag for tag in drill_ticket.tags if tag in self.components]
        found_dc, date = self.parse_infra(drill_ticket)

        if not found_dc:
            logger.info(f"dc not found by drill ticket: {drill_ticket}")
            return

        found_spi = self.get_incidents(drill_ticket, drill_ticket_components)
        for component in drill_ticket_components:
            uchenki_response = self.uchenki_client.post_trainings(
                {
                    'issue': ticket_key,
                    'full_component_name': component,
                    'date': date,
                    'result': [{'dc': found_dc.upper(),
                                'status': UchenkiStatus.not_ok.value if component in found_spi else UchenkiStatus.ok.value}]
                }
            )
            logger.info(f"sending results to uchenki. response:", uchenki_response)

            if uchenki_response.get('message') == 'Done':
                self.set_tag(ticket_key)

    def process_degrade_ticket(self, degrade_ticket):
        ticket_key = degrade_ticket.key
        logger.info(f"start processing degrade ticket: {ticket_key}")
        degrade_ticket_component = ''
        degrade_ticket_handler = ''
        tags = degrade_ticket.tags

        for tag in tags:
            if 'service' in tag:
                degrade_ticket_component = tag.split(':')[1]
            if 'handler' in tag:
                degrade_ticket_handler = tag.split(':')[1]

        if not degrade_ticket_component or not degrade_ticket_handler:
            logger.error(f"cannot understand component and handler from tags {','.join(tags)}")
            return

        # check if both tags exist or no tags exist
        confusing_results = {self.config.degrade_fail_tag, self.config.degrade_success_tag}
        if set(confusing_results).issubset(tags) or not [tag for tag in confusing_results if tag in tags]:
            logger.error(f"cannot understand results from tags {','.join(tags)}")
            return

        _, date = self.parse_infra(degrade_ticket)

        uchenki_response = self.uchenki_client.post_degrade(
            {'issue': ticket_key,
             'full_component_name': degrade_ticket_component,
             'name': degrade_ticket_handler,
             'date': date,
             'result': UchenkiStatus.ok.value if self.config.degrade_success_tag in tags else UchenkiStatus.not_ok.value
             }
        )
        logger.info(f"sending results to uchenki. response:", uchenki_response)
        if uchenki_response.get('message') == 'Done':
            self.set_tag(ticket_key)

    def run_once(self):
        for queue in self.config.queues:
            logger.info(f"start processing queue: {queue}")
            query = f"Queue: {queue} Tags: !{self.config.already_sent_tag} tags: {self.config.drill_ticket_tag}, {self.config.degrade_ticket_tag}"

            for ticket in self.client.issues.find(query):
                tags = ticket.tags

                if self.config.drill_ticket_tag in tags:
                    self.process_drill_ticket(ticket)

                if self.config.degrade_ticket_tag in tags:
                    self.process_degrade_ticket(ticket)
