import datetime
import json

from loguru import logger
from http_clients.abc import AbcClient, Shift
from http_clients.warden import WardenClient, ComponentTag, WardenCheckStatus as CheckStatus, WardenCheckType as CheckType
from startrek_client import Startrek
from jobs import BaseJob
from utils.config import Config
from utils.const import InventoryOptions as IOptions
from utils.templates import render_template
from utils.utils import SetJsonEncoder


class SreInventoryJob(BaseJob):
    def __init__(self, config: Config):
        super().__init__(config)
        self.abc = AbcClient()
        self.warden = WardenClient()
        self.startrek = Startrek(useragent=config.useragent, token=config.st_token)
        self.date_format = "%d-%m-%Y"
        self.last_inventory_check_result = []

    def run_once(self):
        for service_settings in self.config.services:
            validation_dict = {}

            root_component = service_settings['component']
            logger.info(f"start processing root component {root_component}")
            component_data = self.warden.get_component(slug=root_component)
            child_components = self.warden.get_child_components(slug=root_component)

            if IOptions.WARDEN_ALERTS_VALIDATION in service_settings['validations']:
                for alert in component_data.component_checks:

                    if (alert.checkType == CheckType.ALERT
                            and alert.status != CheckStatus.OK and alert.causes.alert_ids):

                        validation_dict.setdefault(IOptions.WARDEN_ALERTS_VALIDATION, {}).setdefault(
                            alert.description, set()).update(set(alert.causes.alert_ids))

            if IOptions.WARDEN_DOCS_VALIDATION in service_settings['validations']:
                for component in child_components:
                    if service_settings['warden_tag'] in component.tags and not component.documentation.objects:
                        validation_dict.setdefault(IOptions.WARDEN_DOCS_VALIDATION, set()).add(component.name)

            self.create_ticket(root_component, service_settings['report_st_queue'], validation_dict)

    def check_new_problems(self, service: str, validation_dict: dict, service_last_ticket_path: str) -> bool:
        logger.info(f"start checking new problems by {service}")
        service_last_ticket_data = json.loads(self.yt.get(service_last_ticket_path))
        current_checks, last_checks = set(validation_dict.keys()), set(service_last_ticket_data.keys())
        is_new_checks = len(current_checks - last_checks) > 0
        logger.info(f'{service}: current checks: {",".join(current_checks)} last checks: {",".join(last_checks)} –– diff: {is_new_checks}')

        if is_new_checks:
            return True

        for check in current_checks.intersection(last_checks):
            current_problems, last_problems = set(validation_dict[check]), set(service_last_ticket_data[check])
            is_new_problems_in_checks = len(current_problems - last_problems) > 0
            logger.info(f'{service}:{check} current problems: {",".join(current_problems)} last problems: {",".join(last_problems)} –– diff: {is_new_problems_in_checks}')

            return is_new_problems_in_checks

    def save_checks_to_yt(self, validation_dict: dict, service_last_ticket_path: str) -> None:
        self.yt.create(
            type=self.string_node_type,
            ignore_existing=True,
            path=service_last_ticket_path,
            recursive=True,
            attributes={"host": self.hostname},
        )

        self.yt.set(service_last_ticket_path, json.dumps(validation_dict, cls=SetJsonEncoder))

    def create_ticket(self, service: str, queue: str, validation_dict: dict) -> None:
        logger.info(f"start creating ticket by {service} with {validation_dict}")
        if not validation_dict:
            return

        service_last_ticket_path = f"{self._locke_path}/{service}"
        service_last_ticket_already_exist = self.yt.exists(service_last_ticket_path)

        if service_last_ticket_already_exist and not self.check_new_problems(service, validation_dict, service_last_ticket_path):
            return
        today = datetime.datetime.now().strftime(self.date_format)
        text = render_template(name="inventory_ticket.jinja2", context={'inventory_dict': validation_dict})
        self.startrek.issues.create(queue=queue,
                                    summary=f"Отчет по инвентаризация сервисов {service} {today}",
                                    description=text,
                                    tags=['sre_inventory'])

        self.save_checks_to_yt(validation_dict, service_last_ticket_path)
