from datetime import datetime, timedelta
from typing import List

from loguru import logger

from http_clients.calendar import CalendarClient
from http_clients.watcher import WatcherClient, Shift
from jobs import BaseJob
from utils.config import Config


class UpdateMeetingOndutyJob(BaseJob):
    def __init__(self, config: Config):
        super().__init__(config)
        self.calendar = CalendarClient()
        self.watcher = WatcherClient()
        self.date_format = "%Y-%m-%d"

    def run_once(self):
        services = self.config.services.as_dict()

        for service_slug, event_id in services.items():
            # если не указывать дату, то календарь вернет ближайшее событие серии
            event = self.calendar.get_event(event_id=event_id)
            logger.info(f"received event from calendar: {event}")
            # получаем дежурных у которых начало дежурства > чем дата события, а конец дежурства < чем дата события
            shifts = self.get_onduty(
                service_slug=service_slug,
                start_date=(event.start_ts + timedelta(days=1)),
                end_date=(event.end_ts - timedelta(days=1)),
            )

            logger.debug(f"for {service_slug=} on date: {event.start_ts} found duty shifts: {shifts}")
            attendees = []

            # проверяем есть ли дежурные за этот день во встрече
            for shift in shifts:
                person = shift.staff
                if person in event.attendees or person in event.optional_attendees or person == event.organizer:
                    logger.info(f"{person} is already in the meeting {event}")
                    continue

                attendees.append(person)

            if not attendees:
                return

            # обновляем участников встречи
            attendees.extend(event.optional_attendees)
            attendees.extend(event.attendees)

            attendees_email = {f"{person.login}@yandex-team.ru" for person in attendees}
            # Чтобы указать переговорки, необходимо указать их почтовые адреса в поле attendees.
            # если не добавить, то переговорка удалится из встречи
            rooms = {resource["email"] for resource in event.resources if resource["resourceType"] == "room"}
            attendees_email.update(rooms)

            logger.info(f"updating event {event}; attendees: {attendees_email}")

            response = self.calendar.update_event(
                event_id=event.id,
                attendees=attendees_email,
                start_ts=event.start_ts,
                end_ts=event.end_ts,
                instance_start_ts=event.instance_start_ts,
            )

            if response.get("status") == "ok":
                logger.info(f"{event} is updated")
            else:
                logger.error(f"event update error! response from calendar: {response}")

    def get_onduty(self, service_slug: str, start_date: datetime, end_date: datetime) -> List[Shift]:
        filter = (
            f"schedule.service.slug={service_slug},"
            f"start__lte={start_date.strftime(self.date_format)},"
            f"end__gte={end_date.strftime(self.date_format)}"
        )

        return self.watcher.shift_list(filter)
