import re
from datetime import datetime, timedelta
from typing import List, Optional, Set

from loguru import logger
from startrek_client import Startrek
from startrek_client.collections import Issues

from http_clients.calendar import CalendarClient, Event
from http_clients.warden import WardenClient
from jobs import BaseJob
from utils.config import Config


class UpdateNanolsrEventJob(BaseJob):
    def __init__(self, config: Config):
        super().__init__(config)
        self.calendar = CalendarClient()
        self.warden = WardenClient()
        self.startrek = Startrek(useragent=config.useragent, token=config.st_token)
        self.date_format = "%Y-%m-%d"

    def run_once(self):
        events_compliance = self.config.events.as_dict()
        now = datetime.now()

        for event_id, components in events_compliance.items():
            incidents = self.get_all_ready_incidents(components)
            for incident in incidents:
                logger.info(f"processing {incident.key}")
                if self.yt.exists(f"{self._locke_path}/{incident.key}"):
                    logger.info(f"{incident.key} has already been processed")
                    continue

                if incident.deadline and datetime.strptime(incident.deadline, self.date_format) > now:
                    # находим нужное событие в календаре, в зависимости от дедлайна
                    event = self.get_nearest_event(event_id, incident.deadline)
                elif not incident.deadline:
                    event = self.get_nearest_event(event_id)
                else:
                    logger.info(f"{incident.key} does not fall under the conditions")
                    continue

                if not event:
                    logger.info(f"not find event for {incident.key}")
                    continue
                logger.info(f"for {incident.key} found {event}")

                if self.add_incident_to_event(incident, event):
                    self.create_node(path=self._locke_path, node_name=incident.key)
                    # TODO: добавлять комментарий про встречу в тикет

    def get_all_ready_incidents(self, components: List[str]) -> List[Issues]:
        all_incidents = []
        for component in components:
            incidents = self.warden.get_incident_list(component_name=component, parent_component_name="")
            for incident in incidents:
                if incident.status == "Разбираем":
                    all_incidents.append(self.startrek.issues[incident.key])
                    logger.info(f"{incident} is ready")

        return all_incidents

    def add_incident_to_event(self, incident: Issues, event: Event) -> bool:
        new_attendees = self.parse_attendees_from_incident(incident)
        attendees = event.get_updated_attendees(new_attendees)

        description = f"https://st.yandex-team.ru/{incident.key}" + "\n" + event.description
        logger.info(
            f"updating {event}; new incident: {incident.key}; attendees: {attendees}; description: {description}"
        )

        response = self.calendar.update_event(
            event_id=event.id,
            attendees=attendees,
            description=description,
            start_ts=event.start_ts,
            instance_start_ts=event.instance_start_ts,
            end_ts=event.end_ts,
        )

        if response.get("status") == "ok":
            logger.info(f"{event} is updated")
            return True
        else:
            logger.error(f"event update error! response from calendar: {response}")
            return False

    def get_nearest_event(self, event_id: int, date: str = None) -> Optional[Event]:
        events = self.finding_events_on_date(event_id, date)
        if not events:
            logger.info(f"not find events for event {event_id} and date {date}")
            return None

        # возвращаем событие для конкретной даты,
        # если она указана то ставим инцидент на разбор независимо от того сколько их уже там
        if date:
            return events[0]

        for event in events:
            # если надо просто вынести инцидент на любой разбор, то ищем ближайшую встречу на которой инцидентов не больше 3х
            if self.count_incidents_on_event(event) < 3:
                return event

    def finding_events_on_date(self, event_id: int, date: str = None) -> List[Event]:
        begin_date = datetime.strptime(date, self.date_format) if date else datetime.now()
        end_date = begin_date + timedelta(days=60)

        return self.calendar.get_events(
            begin_time=begin_date,
            end_time=end_date,
            event_id=event_id,
        )

    def count_incidents_on_event(self, event: Event) -> int:
        return len(re.findall("SPI-\d+", event.description))

    def parse_attendees_from_incident(self, issue: Issues) -> Set[str]:
        block_name = "Кого позвать на разбор инцидента"
        logins = [issue.assignee.id]
        for block in issue.description.split("=== "):
            if block_name.lower() in block.lower():
                # ищем логины вида @login @login
                logins.extend(re.findall("@[\w\-]+", block))

        return set(map(lambda login: f"{login.strip().strip('@')}@yandex-team.ru", logins))
