import datetime
from datetime import timedelta
from typing import List

from loguru import logger

from http_clients.calendar import CalendarClient
from http_clients.warden import WardenClient, Incident
from jobs import BaseJob
from utils.config import Config
from utils.utils import WardenPeriod


class UpdateSpiReviewEventJob(BaseJob):
    def __init__(self, config: Config):
        super().__init__(config)
        self.calendar = CalendarClient()
        self.warden = WardenClient()

    def run_once(self):
        # получаем из конфига соответствие компонент и id встреч в которую добавляем инциденты
        events_compliance = self.config.events.as_dict()

        for event_id, components in events_compliance.items():
            # получаем ближайшую встречу из календаря
            event = self.calendar.get_event(event_id=event_id)
            logger.info(f"processed {event}; components: {components}")
            incidents = self.get_all_incidents_on_week(components=components, event_start_ts=event.start_ts)

            new_incidents = []
            new_attendees = set()
            for incident in incidents:
                if incident.key in event.description:
                    logger.info(f"{incident} was added to the event")
                    continue
                # добавляем ссылку на инцидент, чтобы потом добавить в описание
                new_incidents.append(f"https://st.yandex-team.ru/{incident.key}")
                # и исполнителя в участников встречи
                if incident.assignee:
                    new_attendees.add(f"{incident.assignee}@yandex-team.ru")

            if not new_incidents:
                continue

            logger.info(f"updating {event}; new incidents: {new_incidents}; new attendees: {new_attendees}")
            description = "\n".join(new_incidents) + "\n" + event.description
            attendees = event.get_updated_attendees(new_attendees)

            response = self.calendar.update_event(
                event_id=event.id,
                attendees=attendees,
                description=description,
                start_ts=event.start_ts,
                instance_start_ts=event.instance_start_ts,
                end_ts=event.end_ts,
            )

            if response.get("status") == "ok":
                logger.info(f"{event} is updated")
            else:
                logger.error(f"event update error! response from calendar: {response}")

    def get_all_incidents_on_week(self, components: List[str], event_start_ts: datetime) -> List[Incident]:
        all_incidents = []
        #  получаем все инциденты по всем вертикалям за неделю прошедшую
        #  WardenPeriod.week - возвращает инциденты за неделю от текущей даты, а нам надо за прошедшую неделю до встречи
        #  поэтому приходится брать кастомный период
        created_time = {
            "start": int((event_start_ts - timedelta(days=7)).timestamp()),
            "end": int(event_start_ts.timestamp()),
        }

        for component_name in components:

            incidents = self.warden.get_incident_list(
                component_name=component_name,
                parent_component_name="",
                period=WardenPeriod.custom,
                created_time=created_time,
            )
            logger.debug(f"{component_name} has incidents: {incidents}")
            all_incidents.extend(incidents)

        return all_incidents
