import argparse
import socket
from dataclasses import dataclass
from os import environ as env
from typing import Union, Any

import yaml


@dataclass
class FileLogConfig:
    enabled: bool = False
    path: str = "spihelper.log"
    rotation_rule: str = "100 MB"
    retention_count: int = 5


@dataclass
class StdoutLogConfig:
    enabled: bool = True
    json_formatted: bool = False


@dataclass
class LoggingConfig:
    file: FileLogConfig = FileLogConfig()
    stdout: StdoutLogConfig = StdoutLogConfig()


def _build_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--config', default="config.yaml", help='configuration file in YAML')
    return parser.parse_args()


class ConfigEntry:
    def __getattr__(self, attr: str):
        self.__dict__[attr] = ConfigEntry()

        return ConfigEntry()

    def __bool__(self):
        # return True if len([v for _, v in self.__dict__.items() if not isinstance(v, ConfigEntry)]) > 0 else False
        return any(map(lambda v: not isinstance(v, ConfigEntry), self.__dict__.values()))

    __nonzero__ = __bool__

    def __str__(self) -> str:
        return str(self.as_dict())

    def get(self, item, default: Any = None) -> Union[Any]:
        return getattr(self, item, default)

    def as_dict(self) -> dict:
        return {
            k: v if not isinstance(v, ConfigEntry) else v.as_dict()
            for k, v in self.__dict__.items()
        }


class Config(ConfigEntry):
    logging: LoggingConfig
    abc_token: str
    calendar_token: str
    yt_token: str
    st_token: str
    tvm_token: str
    tvm_tool_url: str
    warden_token: str
    infra_token: str
    uchenki_token: str
    hostname: str
    useragent: str

    def __init__(self, filename: str = ""):
        data = self.load_config(filename=filename)
        data["tvm_token"] = env.get("TVMTOOL_LOCAL_AUTHTOKEN")
        data["tvm_tool_url"] = env.get("DEPLOY_TVM_TOOL_URL")
        data["hostname"] = socket.gethostname()

        for token in ("ABC_TOKEN", "ST_TOKEN", "CALENDAR_TOKEN", "YT_TOKEN", "WARDEN_TOKEN", "INFRA_TOKEN", "UCHENKI_TOKEN"):
            data[token.lower()] = env.get(token)

        def return_or_make_object(o: Any):
            if isinstance(o, dict):
                returning = ConfigEntry()
                for k, v in o.items():
                    setattr(returning, k, return_or_make_object(v))
                return returning
            return o

        for k, v in data.items():
            setattr(self, k, return_or_make_object(v))

    def load_config(self, filename="config.yaml"):
        with open(filename) as f:
            return dict(yaml.safe_load(f))


args = _build_args()
config = Config(filename=args.config)
