import enum
from datetime import datetime
from typing import Optional, List, Tuple

import attr
from attr import has, fields
from cattr import Converter
from cattr.gen import make_dict_unstructure_fn, make_dict_structure_fn, override
from dateutil import parser

import json


class SetJsonEncoder(json.JSONEncoder):
    def default(self, obj):
        if isinstance(obj, set):
            return list(obj)
        return json.JSONEncoder.default(self, obj)


class WardenPeriod(str, enum.Enum):
    last_review = "last_review"
    review = "review"
    last_year = "last_year"
    all = "all"
    week = "week"
    two_weeks = "two_weeks"
    month = "month"
    custom = "custom"


class DCNames(str, enum.Enum):
    IVA = "iva"
    MAN = "man"
    MYT = "myt"
    SAS = "sas"
    VLA = "vla"

    @classmethod
    def list(cls):
        return list(map(lambda c: c.value, cls))


class UchenkiStatus(str, enum.Enum):
    ok = "OK"
    not_ok = "NOT OK"


# конвертер для модели данных из Warden/Calendar - там поля хранятся в camelCase
converter = Converter()


def to_camel_case(snake_str: str) -> str:
    components = snake_str.split("_")
    return components[0] + "".join(x.title() for x in components[1:])


def to_camel_case_unstructure(cls):
    return make_dict_unstructure_fn(
        cls, converter, **{a.name: override(rename=to_camel_case(a.name)) for a in fields(cls)}
    )


def to_camel_case_structure(cls):
    return make_dict_structure_fn(
        cls, converter, **{a.name: override(rename=to_camel_case(a.name)) for a in fields(cls)}
    )


converter.register_unstructure_hook_factory(has, to_camel_case_unstructure)
converter.register_structure_hook_factory(has, to_camel_case_structure)


def str_to_datetime(date_string: str) -> datetime:
    return parser.parse(date_string)


# конвертер для datetime полей attrs
datetime_converter = attr.converters.optional(str_to_datetime)


def get_attrs_class_fields(cls: type, nested_fields: Optional[Tuple] = None) -> List[str]:
    """
    Для attrs классов получаем список атрибутов.
    Если атрибут класса является другим классом, и нужны его поля, то надо передать имена этих аттрибутов в nested_fields
    """
    class_fields = []
    if not nested_fields:
        nested_fields = tuple()

    for field in attr.fields(cls):
        if field.name in nested_fields:
            # вложенные поля получаем как class_name.attrib
            class_fields.extend([f"{field.name}.{_field}" for _field in get_attrs_class_fields(field.type)])
            continue
        class_fields.append(field.name)

    return class_fields
