from .commands import get_all_commands
from .mdb import CloudApi, FolderApi, ClusterApi, ApiAuthProvider
from .utils.http import HTTPError
from .utils.storage import KeyValueStorage
import sys
import getpass


CONFIG_FILENAME = '~/.config/mdb'
CONFIG = None


class CommandDescriptor:
    def __init__(self, cluster, name, args):
        self.cluster = cluster
        self.name = name
        self.args = args


def resolve_cluster_id(string):
    folder_name, cluster_name = string.split('/')
    auth_provider = ApiAuthProvider(CONFIG['token'])
    mdb_cloud = CloudApi(auth_provider)
    clouds = mdb_cloud.list()
    for cloud in clouds:
        if cloud.name == folder_name:
            mdb_folder = FolderApi(auth_provider, cloud)
            folders = mdb_folder.list()
            if len(folders) > 1:
                raise Exception('multiple folders in cluster not supported')
            mdb_cluster = ClusterApi(auth_provider, folders[0])
            for cluster in mdb_cluster.list():
                if cluster.name == cluster_name:
                    return cluster
    return None


def parse_command(args, commands):
    if args and args[0] in commands.keys():
        return CommandDescriptor(None, args[0], args[1:])
    if args and args[1] in commands.keys():
        return CommandDescriptor(resolve_cluster_id(args[0]), args[1], args[2:])
    # fallback to help
    return CommandDescriptor(None, 'help', [])


def main():
    global CONFIG
    CONFIG = KeyValueStorage(CONFIG_FILENAME)
    if 'token' not in CONFIG:
        print("Enter oAuth token for cloud api acccess (https://oauth.yandex-team.ru/authorize?response_type=token&client_id=8cdb2f6a0dca48398c6880312ee2f78d)")
        token = getpass.getpass('Token:')
        CONFIG['token'] = token

    try:
        commands = get_all_commands()
        cmd_descriptor = parse_command(sys.argv[1:], commands)
        cmd = commands[cmd_descriptor.name]
        if cmd_descriptor.cluster:
            cmd.run_for_cluster(cmd_descriptor.cluster, cmd_descriptor.args)
        else:
            cmd.run(cmd_descriptor.args)
    except HTTPError as e:
        print(e.read())
