import json
import logging
import urllib.parse
import requests
import time
import click
import ticket_parser2.api.v1 as tp2

from typing import Optional
from copy import deepcopy

DEFAULT_PS_BILLING_TIMEOUT = 5
DEFAULT_MAGIC_TIMEOUT = 15
DEFAULT_DIRECTORY_TIMEOUT = 15


class BaseApi:
    RETIES = 3
    RETRY_TIMEOUT = 0.100

    def __init__(self, host, service_ticket, timeout):
        self.host = host
        self.session = requests.Session()
        self.session.headers['X-Ya-Service-Ticket'] = service_ticket
        self.session.headers['X-User-Ip'] = '127.0.0.1'
        self.timeout = timeout

    def _make_request(self, method: str, path: str, data: Optional[dict] = None, params: Optional[dict] = None,
                      headers: Optional[dict] = None):
        request_func = getattr(self.session, method.lower())
        request_url = urllib.parse.urljoin(self.host, path)

        for i in range(self.RETIES):
            try:
                logging.info("Making request [%s] %s: headers=%s, params=%s, data=%s",
                             method, request_url, headers, params, data)
                r = request_func(request_url,
                                 headers=headers,
                                 timeout=self.timeout,
                                 params=params,
                                 data=data)
                logging.info("Got response [%s] %s: status=%s", method, request_url, r.status_code)

                r.raise_for_status()
                return r.json()
            except requests.RequestException as e:
                if e.response:
                    logging.error(
                        "Failed to make request [%s] %s: headers=%s, params=%s, data=%s, status=%s, response=%s",
                        method, request_url, headers, params, data, e.response.status_code, e.response.text
                    )
                else:
                    logging.error(
                        "Failed to make request [%s] %s: headers=%s, params=%s, data=%s, error=%s",
                        method, request_url, headers, params, data, e
                    )
            except ValueError:
                logging.exception(
                    "Failed to make request [%s] %s: headers=%s, params=%s, data=%s, status=%s, response=%s",
                    method, request_url, headers, params, data
                )

            if i < self.RETIES - 1:
                logging.error(
                    "Retrying request  [%s] %s: headers=%s, params=%s, data=%s",
                    method, request_url, headers, params, data
                )
                time.sleep(self.RETRY_TIMEOUT)
            else:
                logging.error(
                    "Retry limit exceeded on [%s] %s: headers=%s, params=%s, data=%s",
                    method, request_url, headers, params, data
                )

        return {}


class MagicAPI(BaseApi):
    def get_user_info(self, login) -> Optional[dict]:
        return self._make_request('POST', '/api/external/userinfo', data={'login': login}).get('result')

    def get_settings(self, login):
        return self._make_request('POST', '/api/external/settings', data={'login': login}).get('result')

    def get_pro_products(self, login):
        return self._make_request('GET', '/api/external/pro',
                                  data={'login': login}).get('result', {}).get('pro_products')

class PSBillingAPI(BaseApi):
    def get_features_for_admin(self, uid):
        return self._make_request('GET', f'/v1/groups/organization/admins/{uid}/features', params={'service': 'mail'})


class DirectoryAPI(BaseApi):
    def admin_organizations(self, uid):
        return self._make_request('GET', '/v11/organizations/', headers={'X-UID': str(uid)},
                                  params={'admin_only': 'true'}).get('result')


@click.command()
@click.option('--input-json', required=True, type=str)
@click.option('--output-json', required=True, type=str)
@click.option('--tvm-client-id', required=True, type=int)
@click.option('--tvm-secret', required=True, type=str)
@click.option('--ps-billing-host', required=True, type=str)
@click.option('--ps-billing-client-id', required=True, type=int)
@click.option('--ps-billing-timeout', type=float, default=DEFAULT_PS_BILLING_TIMEOUT)
@click.option('--magic-host', required=True, type=str)
@click.option('--magic-client-id', required=True, type=int)
@click.option('--magic-timeout', type=float, default=DEFAULT_MAGIC_TIMEOUT)
@click.option('--directory-host', required=True, type=str)
@click.option('--directory-client-id', required=True, type=int)
@click.option('--directory-timeout', type=float, default=DEFAULT_DIRECTORY_TIMEOUT)
@click.option('--verbose', type=bool, is_flag=True)
def run(input_json, output_json, tvm_client_id, tvm_secret, verbose,
        ps_billing_host, ps_billing_client_id, ps_billing_timeout,
        magic_host, magic_client_id, magic_timeout,
        directory_host, directory_client_id, directory_timeout):
    """
    Nirvana operation for retrieving information about priority support of Yandex.Mail user
    For more details: https://st.yandex-team.ru/PDDMAIL-468
    """
    logging.basicConfig(format='%(asctime)s - %(name)s - %(levelname)s - %(message)s')
    if verbose:
        logging.root.setLevel(logging.INFO)

    tvm_client = tp2.TvmClient(tp2.TvmApiClientSettings(
        self_client_id=tvm_client_id,
        self_secret=tvm_secret,
        dsts={"magic": magic_client_id, "ps-billing": ps_billing_client_id, 'directory': directory_client_id},
    ))

    magic_api = MagicAPI(magic_host, tvm_client.get_service_ticket_for("magic"), magic_timeout)
    ps_billing_api = PSBillingAPI(ps_billing_host, tvm_client.get_service_ticket_for("ps-billing"), ps_billing_timeout)
    directory_api = DirectoryAPI(directory_host, tvm_client.get_service_ticket_for("directory"), directory_timeout)

    def enrich_data(data):
        r = deepcopy(data)
        r['priority_support_admin'] = False
        r['priority_support_user'] = False
        r['connect_admin'] = False
        r['has_mail_pro'] = False

        try:
            if 'user' not in r or 'login' not in r['user']:
                logging.error('Got ticket without login: %s', r)
                return r

            login = r['user']['login']
            user_info = magic_api.get_user_info(login)

            if not user_info or 'uid' not in user_info:
                logging.error('Failed to get user_info for %s', login)
                return r

            settings = magic_api.get_settings(login)
            try:
                has_priority_support = settings['settings']['parameters']['single_settings'].get('has_priority_support')
                r['priority_support_user'] = has_priority_support == 'on'
            except (KeyError, TypeError):
                logging.error("Failed to get settings for user: %s", login)

            features = ps_billing_api.get_features_for_admin(user_info['uid'])
            try:
                r['priority_support_admin'] = features.get('mail_priority_support', False)
            except TypeError:
                logging.error("Failed to get psbilling features for user: %s", login)

            organizations = directory_api.admin_organizations(user_info['uid'])
            if organizations and isinstance(organizations, list):
                r['connect_admin'] = len(organizations) > 0

            pro_products = magic_api.get_pro_products(login)
            if pro_products:
                r['has_mail_pro'] = True
        except:
            logging.exception("Got error processing %s", data)

        return r

    with open(input_json) as f:
        input_data = json.load(f)

    result = [enrich_data(d) for d in input_data]

    with open(output_json, 'w') as f:
        json.dump(result, f)


if __name__ == '__main__':
    run()
