package cmd

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"

	"github.com/spf13/cobra"
)

type YcHost struct {
	Health string `json:"health"`
	Name   string `json:"name"`
	ZoneID string `json:"zoneId"`
	Role   string `json:"role"`
}

type YcCluster struct {
	Name string `json:"name"`
}

type YcListHosts struct {
	Hosts []YcHost `json:"hosts"`
}

func getIAMToken(client *http.Client) string {
	type IAMToken struct {
		Token     string `json:"iamToken"`
		ExpiresAt string `json:"expiresAt"`
	}

	if cli.OAuthToken == "" {
		log.Fatal("OAuthToken is not set, exit")
	}

	url := fmt.Sprintf("%s/v1/tokens", cli.YcIAMAPIURL)
	body := bytes.NewBuffer([]byte(`{"yandexPassportOauthToken":"` + cli.OAuthToken + `"}`))

	req, err := http.NewRequest("POST", url, body)
	if err != nil {
		log.Fatal(err)
	}

	req.Header.Set("Content-Type", "application/json")

	resp, err := client.Do(req)
	if err != nil {
		log.Fatal(err)
	} else if resp.StatusCode != 200 {
		bodyBytes, _ := ioutil.ReadAll(resp.Body)
		log.Fatal(fmt.Sprintf("Url %s ", url), string(bodyBytes))
	}

	token := IAMToken{}
	err = json.NewDecoder(resp.Body).Decode(&token)
	if err != nil {
		log.Fatal(err)
	}

	err = resp.Body.Close()
	if err != nil {
		log.Fatal(err)
	}

	return token.Token
}

var dbCmd = &cobra.Command{
	Use:   "db",
	Short: "Show db clusters state",
	RunE: func(cmd *cobra.Command, args []string) error {
		progress, bar := cli.CreateProgressBar(len(cli.Config.Yc.Clusters), "")
		client := &http.Client{}

		iamToken := getIAMToken(client)

		for _, cluster := range cli.Config.Yc.Clusters {
			url := fmt.Sprintf(
				"%s/managed-%s/v1/clusters/%s",
				cli.YcAPIURL,
				cluster.Type,
				cluster.ID,
			)
			urlHosts := url + "/hosts"

			// Cluster
			req, err := http.NewRequest("GET", url, nil)
			if err != nil {
				log.Fatal(err)
			}
			req.Header.Add("Authorization", "Bearer "+iamToken)
			resp, err := client.Do(req)
			if err != nil {
				log.Fatal(err)
			} else if resp.StatusCode != 200 {
				bodyBytes, _ := ioutil.ReadAll(resp.Body)
				log.Fatal(fmt.Sprintf("Url %s ", url), string(bodyBytes))
			}

			var ycCluster YcCluster
			err = json.NewDecoder(resp.Body).Decode(&ycCluster)
			if err != nil {
				log.Fatal(err)
			}
			err = resp.Body.Close()
			if err != nil {
				log.Fatal(err)
			}

			// Hosts
			req, err = http.NewRequest("GET", urlHosts, nil)
			if err != nil {
				log.Fatal(err)
			}
			req.Header.Add("Authorization", "Bearer "+iamToken)
			resp, err = client.Do(req)
			if err != nil {
				log.Fatal(err)
			} else if resp.StatusCode != 200 {
				bodyBytes, _ := ioutil.ReadAll(resp.Body)
				log.Fatal(fmt.Sprintf("Url %s ", url), string(bodyBytes))
			}

			var ycListHosts YcListHosts
			err = json.NewDecoder(resp.Body).Decode(&ycListHosts)
			if err != nil {
				log.Fatal(err)
			}
			err = resp.Body.Close()
			if err != nil {
				log.Fatal(err)
			}

			log.Printf("Cluster %s (%s)\n", ycCluster.Name, cluster.ID)
			for _, host := range ycListHosts.Hosts {
				log.Printf("Name: %s, role: %s, zone: %s, health: %s\n", host.Name, host.Role, host.ZoneID, host.Health)
			}
			log.Println()

			bar.Increment()
		}
		progress.Wait()
		return nil
	},
}

func init() {
	cmd := dbCmd
	rootCmd.AddCommand(cmd)
}
