package cmd

import (
	"fmt"
	"log"
	"os"
	"text/tabwriter"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/mail/swat/dutytool/pkg/ctl"
	"a.yandex-team.ru/mail/swat/dutytool/pkg/interactions"
)

type Resourses struct {
	CPU     uint64
	Mem     uint64
	Disk    uint64
	DiskBW  uint64
	Network uint64
}

type Host struct {
	ComponentID string
	Host        string
	IPAddress   string

	Resources Resourses
	Usage     Resourses
}

func getDeployStageHosts(
	dc string,
	stage ctl.DeployStage,
	deployClient interactions.DeployClient,
	solomonClient interactions.SolomonClient,
	collectUsageStats bool,
) []Host {
	hosts := make([]Host, 0)
	deployStage, err := deployClient.GetStage(stage.Name)
	if err != nil {
		log.Fatalln(err)
	}
	for _, unit := range deployStage.DeployUnits {
		duHosts := make([]*Host, 0)

		for _, replicaSet := range unit.ReplicaSetsInDC(interactions.YPDataCenter(dc)) {
			podSetID := replicaSet.ID // Хакерство? Я не выкупил, как получить список PodSet у ReplicaSet. Но ID у них на практике одинаковые
			pods, err := deployClient.GetPods(podSetID, interactions.YPDataCenter(dc))
			if err != nil {
				log.Fatalln(err)
			}

			for _, pod := range pods {
				item := Host{
					ComponentID: pod.ID,
					Host:        pod.Host,
					IPAddress:   pod.IP,
				}
				item.Resources.CPU = unit.Resourses.CPU
				item.Resources.Mem = unit.Resourses.Mem
				item.Resources.Disk = unit.Resourses.Disk
				item.Resources.DiskBW = unit.Resourses.DiskBW
				item.Resources.Network = unit.Resourses.Network

				duHosts = append(duHosts, &item)
			}
		}

		if collectUsageStats {
			// CPU usage
			resp, err := solomonClient.GetMetrics(
				"yasm_deploy",
				time.Now().Add(-time.Hour*24),
				time.Now(),
				fmt.Sprintf(`histogram_percentile(99, '', {project='yasm_deploy',geo='%s',stage='%s',deploy_unit='%s',hosts='ASEARCH',signal='portoinst-cpu_usage_slot_hgram'})`, dc, stage.Name, unit.ID),
			)
			if err != nil {
				log.Fatalln(err)
			}

			for _, host := range duHosts {
				host.Usage.CPU = uint64(resp.Max(1024) * 1000)
			}

			// RAM usage
			resp, err = solomonClient.GetMetrics(
				"yasm_deploy",
				time.Now().Add(-time.Hour*24),
				time.Now(),
				fmt.Sprintf(`histogram_percentile(99, '', {project='yasm_deploy',geo='%s',stage='%s',deploy_unit='%s',hosts='ASEARCH',signal='portoinst-anon_usage_slot_hgram'})`, dc, stage.Name, unit.ID),
			)

			if err != nil {
				log.Fatalln(err)
			}

			for _, host := range duHosts {
				host.Usage.Mem = uint64(resp.Max(1024))
			}

			// SSD usage
			usageFrac := getDiskUsage(solomonClient, dc, stage, unit)
			for _, host := range duHosts {
				host.Usage.Disk = uint64(float64(host.Resources.Disk) * usageFrac)
			}

			// Network usage
			resp, err = solomonClient.GetMetrics(
				"yasm_deploy",
				time.Now().Add(-time.Hour*24),
				time.Now(),
				fmt.Sprintf(`histogram_percentile(100, '', {project='yasm_deploy',geo='%s',stage='%s',deploy_unit='%s',hosts='ASEARCH',signal='portoinst-net_rx_utilization_hgram'})`, dc, stage.Name, unit.ID),
			)

			if err != nil {
				log.Fatalln(err)
			}

			for _, host := range duHosts {
				host.Usage.Network = uint64(resp.Max(1024))
			}

		}

		for _, duHost := range duHosts {
			hosts = append(hosts, *duHost)
		}

	}
	return hosts
}

// getDiskUsage returns Deploy unit usage of disk quota as a fraction [0.0, ...]
// Deploy names for the main disk can vary, as they are user-defined.
// For example, main disk name for oplata stages is 'infra', because of transfer from Qloud.
func getDiskUsage(
	solomonClient interactions.SolomonClient,
	dc string,
	stage ctl.DeployStage,
	deployUnit interactions.DeployUnit,
) float64 {

	signals := []string{"main-disk_hgram", "disk-0_hgram", "infra_hgram"}

	for _, signal := range signals {
		resp, err := solomonClient.GetMetrics(
			"yasm_deploy",
			time.Now().Add(-time.Hour*24),
			time.Now(),
			fmt.Sprintf(`histogram_percentile(99, '', {project='yasm_deploy',geo='%s',stage='%s',deploy_unit='%s',hosts='ASEARCH',signal='portoinst-capacity-perc_usage_/virtual_disks/%s'})`, dc, stage.Name, deployUnit.ID, signal),
		)

		if err != nil {
			log.Fatalln(err)
		}

		if len(resp.Vector) != 0 {
			return resp.Max(1024) / 100
		}
	}

	return 0

}

func getQloudEnvironmentHosts(
	dc string,
	project ctl.QloudProject,
	application ctl.QloudApplication,
	environment ctl.QloudEnvironment,
	qloudClient interactions.QloudClient,
) []Host {
	hosts := make([]Host, 0)

	qloudEnv, err := qloudClient.GetEnvironment(project.Name, application.Name, environment.Name)
	if err != nil {
		log.Fatal(err)
	}

	for _, component := range qloudEnv.Components {
		for _, instance := range component.InstancesInDC(dc) {
			item := Host{
				ComponentID: component.Event.ComponentID,
				Host:        instance.Host,
				IPAddress:   instance.InstanceIP,
			}
			hosts = append(hosts, item)
		}
	}
	return hosts
}

// toKb converts bytes to Kilobytes.
func toKb(bytes uint64) float64 {
	return float64(bytes) / (1 << 10)
}

// toMb converts bytes to Megabytes.
func toMb(bytes uint64) float64 {
	return float64(bytes) / (1 << 20)
}

// toGb converts bytes to Gigabytes.
func toGb(bytes uint64) float64 {
	return float64(bytes) / (1 << 30)
}

// usageStringWithLimit returns string of format `<usage> of <limit> (<usage_perc>%)` and if limit is not defined(==0)
// returns message, that limit is not defined.
func usageStringWithLimit(usage uint64, limit uint64) string {
	var infoString string

	if limit != 0 {
		infoString = fmt.Sprintf("%d of %d (%.2f%%)", usage, limit, float64(usage)/float64(limit)*100)
	} else {
		infoString = fmt.Sprintf("%d of 0 (limit is not set)", usage)
	}

	return infoString
}

// usageStringWithLimitF returns string of format `<usage> of <limit> (<usage_perc>%)` and if limit is not defined(==0)
// returns message, that limit is not defined.mat
func usageStringWithLimitF(usage float64, limit float64) string {
	var infoString string

	if limit != 0 {
		infoString = fmt.Sprintf("%.2f of %.2f (%.2f%%)", usage, limit, usage/limit*100)
	} else {
		infoString = fmt.Sprintf("%.2f of 0 (limit is not set)", usage)
	}

	return infoString
}

var dcHostsCmd = &cobra.Command{
	Use:   "dc-hosts",
	Short: "Show instances in selected DC",
	RunE: func(cmd *cobra.Command, args []string) error {
		dc, err := cmd.Flags().GetString("dc")
		if err != nil {
			return err
		}

		usageStats, err := cmd.Flags().GetBool("usage-stats")
		if err != nil {
			return err
		}

		hosts := make([]Host, 0)

		qloudClient := interactions.CreateQloudClient(*cli)
		solomonClient := interactions.CreateSolomonClient(*cli)
		deployClient := interactions.CreateDeployClient(*cli)
		defer deployClient.Close()

		amountEnvs := cli.Config.Qloud.AmountEnvs() + cli.Config.Deploy.AmountStages()

		if amountEnvs == 0 {
			log.Println("Nothing to analyze, skipping")
			return nil
		}

		progress, bar := cli.CreateProgressBar(amountEnvs, "")

		for _, project := range cli.Config.Deploy.Projects {
			for _, stage := range project.Stages {
				deployHosts := getDeployStageHosts(dc, stage, deployClient, solomonClient, usageStats)
				hosts = append(hosts, deployHosts...)
				bar.Increment()
			}
		}

		for _, project := range cli.Config.Qloud.Projects {
			for _, application := range project.Applications {
				for _, environment := range application.Environments {
					qloudHosts := getQloudEnvironmentHosts(dc, project, application, environment, qloudClient)
					hosts = append(hosts, qloudHosts...)
					bar.Increment()
				}
			}
		}

		progress.Wait()

		writer := new(tabwriter.Writer)
		writer.Init(os.Stdout, 4, 4, 4, ' ', 0)

		// write header
		headerMask := ""
		headerItems := []interface{}{"Component", "Host", "IP"}
		cols := 3
		if usageStats {
			cols += 5
			headerItems = append(headerItems, "VCPU", "Memory (Gb)", "Disk (Gb)", "Disk BW (Mb)", "Network")
		}
		for i := 0; i < cols; i += 1 {
			headerMask += "%s\t"
		}
		headerMask += "\n"

		if _, err := fmt.Fprintf(writer, headerMask, headerItems...); err != nil {
			log.Fatal(err)
		}

		for _, item := range hosts {
			mask := "%s\t%s\t%s\t"
			items := []interface{}{item.ComponentID, item.Host, item.IPAddress}
			if usageStats {
				mask += "%s\t%s\t%s\t%.2f\t%s\t"

				items = append(
					items,
					usageStringWithLimit(item.Usage.CPU, item.Resources.CPU),
					usageStringWithLimitF(toGb(item.Usage.Mem), toGb(item.Resources.Mem)),
					usageStringWithLimitF(toGb(item.Usage.Disk), toGb(item.Resources.Disk)),
					toMb(item.Resources.DiskBW),
					usageStringWithLimit(item.Usage.Network, item.Resources.Network),
				)
			}
			mask += "\n"

			_, err := fmt.Fprintf(writer, mask, items...)
			if err != nil {
				log.Fatal(err)
			}
		}

		err = writer.Flush()
		if err != nil {
			log.Fatal(err)
		}

		return nil
	},
}

func init() {
	cmd := dcHostsCmd
	cmd.Flags().String("dc", "", "DC (vla, iva, man, sas, myt)")
	cmd.Flags().Bool("usage-stats", false, "Show usage stat")
	ctl.RequiredFlags(cmd.Flags(), "dc")
	rootCmd.AddCommand(cmd)
}
