package cmd

import (
	"bytes"
	"errors"
	"fmt"
	"log"
	"strconv"
	"strings"
	"text/template"

	"github.com/mitchellh/mapstructure"
	"github.com/spf13/cobra"
	"github.com/spf13/pflag"

	"a.yandex-team.ru/mail/swat/dutytool/pkg/ctl"
	"a.yandex-team.ru/mail/swat/dutytool/pkg/interactions"
)

type DrillCmdType int

const (
	DrillCmdTypeQloud  DrillCmdType = iota
	DrillCmdTypeDeploy DrillCmdType = iota
)

type PlanOpts struct {
	Format          string
	User            string
	InteractiveMode bool
	CmdsOpen        bool
	CmdsClose       bool
}

func (opts *PlanOpts) loadFromFlags(flags *pflag.FlagSet) {
	opts.InteractiveMode, _ = flags.GetBool("not-interactive")
	opts.InteractiveMode = !opts.InteractiveMode
	opts.User, _ = flags.GetString("user")
	opts.CmdsOpen, _ = flags.GetBool("open")
	opts.CmdsClose, _ = flags.GetBool("close")

	conductorFormat, _ := flags.GetBool("conductor")
	if conductorFormat {
		opts.Format = "conductor"
	} else {
		opts.Format = "qloud"
	}

	if opts.CmdsOpen == opts.CmdsClose && opts.CmdsOpen {
		log.Fatal("--open and --close can't be used simultaneously")
	}

	if !opts.InteractiveMode && opts.User == "" {
		log.Fatal("--user required in --non-interactive mode")
	}
}

type DrillCmd struct {
	Components []*string
	Cmd        string
	Type       DrillCmdType
}

func (drillCmd DrillCmd) toString(dc string, planOpts PlanOpts) string {
	var hostPrefix string
	user := planOpts.User

	if drillCmd.Type == DrillCmdTypeQloud {
		if planOpts.Format == "conductor" {
			hostPrefix = "%"
		} else {
			hostPrefix = "q:"
		}
	} else {
		hostPrefix = "d:"
		user = "root"
	}

	var cmd string
	if planOpts.InteractiveMode {
		cmd = drillCmd.Cmd
	} else {
		cmd = escapeShellCmd(drillCmd.Cmd)
	}

	tmplRaw := "p_exec {{range $i, $component := .Components}}{{if gt $i 0}},{{end}}{{$.Prefix}}{{$component}}@{{$.DC}}{{end}} {{$.Cmd}}"
	if !planOpts.InteractiveMode {
		tmplRaw = "executer -u {{$.User}} --cached " + tmplRaw
	}

	var result bytes.Buffer
	context := struct {
		Components []*string
		Prefix     string
		DC         string
		Cmd        string
		User       string
	}{
		Components: drillCmd.Components,
		DC:         dc,
		Prefix:     hostPrefix,
		Cmd:        cmd,
		User:       user,
	}

	tmpl, _ := template.New("drillCmd").Parse(tmplRaw)
	err := tmpl.Execute(&result, context)
	if err != nil {
		log.Fatalln(err)
	}

	return result.String()
}

type DrillPlanItem struct {
	Project   string
	Comments  []string
	CloseCmds map[string]*DrillCmd
	OpenCmds  map[string]*DrillCmd
}

func escapeShellCmd(cmd string) string {
	return "\"" + strings.ReplaceAll(cmd, "\"", "\\\"") + "\""
}

func parseDrills(drills interface{}) (hasDrills bool, drillsOut []ctl.Drill, err error) {
	if drillsBool, ok := drills.(bool); ok {
		if drillsBool {
			err = errors.New("drills should be false or array of Drills")
		} else {
			hasDrills = false
		}
	} else if drillsIterfaces, ok := drills.([]interface{}); ok {
		hasDrills = true
		err = mapstructure.Decode(drillsIterfaces, &drillsOut)
	} else if drills != nil {
		err = errors.New("unknown drill type")
	}
	return
}

func generateDeployStagePlan(
	dc string,
	project ctl.DeployProject,
	stage ctl.DeployStage,
	drillPlan map[string]*DrillPlanItem,
	deployClient interactions.DeployClient,
) (missedComponents []string) {
	missedComponents = make([]string, 0)

	var drills []ctl.Drill
	var err error
	var hasDrills bool

	hasDrills, drills, err = parseDrills(stage.Drills)

	if err != nil {
		log.Fatalln(err)
	}
	if !hasDrills {
		return
	}

	deployStage, err := deployClient.GetStage(stage.Name)
	if err != nil {
		log.Fatalln(err)
	}

	for _, unit := range deployStage.DeployUnits {
		if len(unit.ReplicaSetsInDC(interactions.YPDataCenter(dc))) > 0 {
			hasMatch := false
			for index, drill := range drills {
				if drill.HasComponent(unit.ID) {
					hasMatch = true
					if !drill.Skip {
						projectName := project.Name
						cmdKey := stage.Name + "-" + strconv.Itoa(index)
						componentName := unit.Fullname()

						if _, ok := drillPlan[projectName]; !ok {
							drillPlan[projectName] = &DrillPlanItem{
								Project:   projectName,
								OpenCmds:  map[string]*DrillCmd{},
								CloseCmds: map[string]*DrillCmd{},
							}
						}
						drillPlan[projectName].Comments = append(
							drillPlan[projectName].Comments,
							drill.Comment,
						)

						if _, ok := drillPlan[projectName].OpenCmds[cmdKey]; !ok {
							drillPlan[projectName].OpenCmds[cmdKey] = &DrillCmd{Cmd: drill.CmdOpen, Type: DrillCmdTypeDeploy}
						}
						drillPlan[projectName].OpenCmds[cmdKey].Components = append(
							drillPlan[projectName].OpenCmds[cmdKey].Components,
							&componentName,
						)

						if _, ok := drillPlan[projectName].CloseCmds[cmdKey]; !ok {
							drillPlan[projectName].CloseCmds[cmdKey] = &DrillCmd{Cmd: drill.CmdClose, Type: DrillCmdTypeDeploy}
						}
						drillPlan[projectName].CloseCmds[cmdKey].Components = append(
							drillPlan[projectName].CloseCmds[cmdKey].Components,
							&componentName,
						)
					}
				}
			}

			if !hasMatch {
				missedComponents = append(missedComponents, unit.Fullname())
			}
		}
	}
	return
}

func generateQloudEnvironmentPlan(
	dc string,
	project ctl.QloudProject,
	application ctl.QloudApplication,
	environment ctl.QloudEnvironment,
	drillPlan map[string]*DrillPlanItem,
	planOpts PlanOpts,
	qloudClient interactions.QloudClient,
) (missedComponents []string) {
	missedComponents = make([]string, 0)

	var drills []ctl.Drill
	var err error
	var hasDrills bool

	hasDrills, drills, err = parseDrills(environment.Drills)

	if err != nil {
		log.Fatalln(err)
	}
	if !hasDrills {
		return
	}

	qloudEnv, err := qloudClient.GetEnvironment(project.Name, application.Name, environment.Name)
	if err != nil {
		log.Fatalln(err)
	}

	for _, component := range qloudEnv.Components {
		if len(component.InstancesInDC(dc)) > 0 {
			hasMatch := false
			for index, drill := range drills {
				if drill.HasComponent(component.Name) {
					hasMatch = true
					if !drill.Skip {
						projectName := project.Name + "." + application.Name
						cmdKey := environment.Name + "-" + strconv.Itoa(index)
						componentName := component.Fullname(planOpts.Format)

						if _, ok := drillPlan[projectName]; !ok {
							drillPlan[projectName] = &DrillPlanItem{
								Project:   projectName,
								OpenCmds:  map[string]*DrillCmd{},
								CloseCmds: map[string]*DrillCmd{},
							}
						}
						drillPlan[projectName].Comments = append(
							drillPlan[projectName].Comments,
							drill.Comment,
						)

						if _, ok := drillPlan[projectName].OpenCmds[cmdKey]; !ok {
							drillPlan[projectName].OpenCmds[cmdKey] = &DrillCmd{Cmd: drill.CmdOpen, Type: DrillCmdTypeQloud}
						}
						drillPlan[projectName].OpenCmds[cmdKey].Components = append(
							drillPlan[projectName].OpenCmds[cmdKey].Components,
							&componentName,
						)

						if _, ok := drillPlan[projectName].CloseCmds[cmdKey]; !ok {
							drillPlan[projectName].CloseCmds[cmdKey] = &DrillCmd{Cmd: drill.CmdClose, Type: DrillCmdTypeQloud}
						}
						drillPlan[projectName].CloseCmds[cmdKey].Components = append(
							drillPlan[projectName].CloseCmds[cmdKey].Components,
							&componentName,
						)
					}
				}
			}

			if !hasMatch {
				missedComponents = append(missedComponents, component.Fullname("qloud"))
			}
		}
	}
	return
}

func generatePlan(dc string, planOpts PlanOpts) string {
	amountEnvs := cli.Config.Qloud.AmountEnvs() + cli.Config.Deploy.AmountStages()

	qloudClient := interactions.CreateQloudClient(*cli)
	deployClient := interactions.CreateDeployClient(*cli)
	defer deployClient.Close()

	if amountEnvs == 0 {
		log.Fatal("No components, exit.")
	}

	progress, bar := cli.CreateProgressBar(amountEnvs, dc+": ")
	drillPlan := map[string]*DrillPlanItem{}
	missedComponents := make([]string, 0)

	for _, project := range cli.Config.Deploy.Projects {
		for _, stage := range project.Stages {
			missedDeployComponents := generateDeployStagePlan(dc, project, stage, drillPlan, deployClient)
			missedComponents = append(missedComponents, missedDeployComponents...)
			bar.Increment()
		}
	}

	for _, project := range cli.Config.Qloud.Projects {
		for _, application := range project.Applications {
			for _, environment := range application.Environments {
				missedDeployComponents := generateQloudEnvironmentPlan(dc, project, application, environment, drillPlan, planOpts, qloudClient)
				missedComponents = append(missedComponents, missedDeployComponents...)
				bar.Increment()
			}
		}
	}
	progress.Wait()

	var tmplRaw string

	if planOpts.CmdsOpen || planOpts.CmdsClose {
		mode := "Open"
		if planOpts.CmdsClose {
			mode = "Close"
		}
		tmplRaw = fmt.Sprintf(`{{range .Items}}{{range .%sCmds}}{{.}}%s{{end}}{{end}}`, mode, "\n")
	} else {
		tmplRaw = `
===План===
#|
|| **Проект** | **Выключение** | **Включение** | **Время [выкл] [вкл]** | **Заметки** ||
{{range .Items}}
||
{{.Project}} |
%%(bash nomark){{range .CloseCmds}}{{.}}
{{end}}%% |
%%(bash nomark){{range .OpenCmds}}{{.}}
{{end}}%% |
%%[00:00 00:00]%% |
{{range .Comments}}{{.}}{{end}}
||
{{end}}
|#

{{if .HaveMissedComponents}}===Пропущенные компоненты===
{{range .MissedComponents}}* {{ . }}
{{end}}{{end}}
`
	}

	type RenderItem struct {
		Project   string
		Comments  []string
		OpenCmds  []string
		CloseCmds []string
	}

	items := make([]RenderItem, 0)
	for _, v := range drillPlan {
		openCmds := make([]string, 0)
		closeCmds := make([]string, 0)

		for _, cmd := range v.OpenCmds {
			openCmds = append(openCmds, cmd.toString(dc, planOpts))
		}

		for _, cmd := range v.CloseCmds {
			closeCmds = append(closeCmds, cmd.toString(dc, planOpts))
		}

		items = append(items, RenderItem{
			Project:   v.Project,
			Comments:  v.Comments,
			OpenCmds:  openCmds,
			CloseCmds: closeCmds,
		})
	}

	var result bytes.Buffer
	context := struct {
		Items                []RenderItem
		MissedComponents     []string
		HaveMissedComponents bool
	}{
		Items:                items,
		MissedComponents:     missedComponents,
		HaveMissedComponents: len(missedComponents) > 0,
	}
	tmpl, _ := template.New("drills").Parse(tmplRaw)
	err := tmpl.Execute(&result, context)
	if err != nil {
		log.Fatalln(err)
	}

	return strings.Trim(result.String(), "\n")
}

var drillsCmd = &cobra.Command{
	Use:   "drills",
	Short: "Generate drills plan in wiki format",
	RunE: func(cmd *cobra.Command, args []string) error {
		dcs := []string{"vla", "iva", "myt", "sas", "man"}
		dcFilter, _ := cmd.Flags().GetString("dc")

		planOps := PlanOpts{}
		planOps.loadFromFlags(cmd.Flags())

		for _, dc := range dcs {
			if dcFilter == "" || dc == dcFilter {
				fmt.Println(generatePlan(dc, planOps))
			}
		}

		return nil
	},
}

func init() {
	cmd := drillsCmd
	cmd.Flags().String("dc", "", "DC (vla, iva, man, sas, myt)")
	cmd.Flags().Bool("conductor", false, "Generate executor cmd in conductor format")
	cmd.Flags().Bool("not-interactive", false, "For not-interactive executor mode")
	cmd.Flags().String("user", "", "Executer user")
	cmd.Flags().Bool("open", false, "Output cmds for services opening")
	cmd.Flags().Bool("close", false, "Output cmds for services closing")

	rootCmd.AddCommand(cmd)
}
