package cmd

import (
	"context"
	"encoding/json"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
	"path"
	"text/template"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
	"a.yandex-team.ru/mail/swat/dutytool/pkg/ctl"
)

func getSecret(secretID string, versionID string) (map[string]string, error) {
	yav, err := httpyav.NewClient(
		httpyav.WithOAuthToken(cli.OAuthToken),
	)
	if err != nil {
		return nil, err
	}

	resp, err := yav.GetVersion(context.Background(), versionID)
	if err != nil {
		return nil, err
	}

	secret := make(map[string]string)
	for _, value := range resp.Version.Values {
		secret[value.Key] = value.Value
	}
	return secret, nil
}

func getAbsoluteTemplatePath(templateArcadiaPath string) (string, error) {
	cmd := exec.Command("ya", "dump", "root")

	output, err := cmd.Output()
	if err != nil {
		return "", err
	}
	return path.Join(string(output), templateArcadiaPath), nil
}

var deployTemplateCmd = &cobra.Command{
	Use:   "yadeploy-template",
	Short: "Renders deploy stage spec from the template",
	Long:  "Serves for substituting delegation token variables of yadeploy stage spec with actual values.",
	RunE: func(cmd *cobra.Command, args []string) error {
		stageID, err := cmd.Flags().GetString("stage-id")
		if err != nil {
			return err
		}

		var targetStage ctl.DeployStage
		for _, project := range cli.Config.Deploy.Projects {
			for _, stage := range project.Stages {
				if stage.Name == stageID {
					targetStage = stage
				}
			}
		}
		if targetStage.Name == "" {
			log.Fatalf("Stage %s not found", stageID)
		}

		stageSpec := targetStage.StageSpec
		if stageSpec.TemplateArcadiaPath == "" {
			log.Fatalf("stage_spec.template_arcadia_path is not set")
		}

		templatePath, err := getAbsoluteTemplatePath(stageSpec.TemplateArcadiaPath)
		if err != nil {
			log.Fatal(err)
		}

		tmplRaw, err := ioutil.ReadFile(templatePath)
		if err != nil {
			log.Fatal(err)
		}

		tmpl, err := template.New("yadeploy-stage-spec").Parse(string(tmplRaw))
		if err != nil {
			log.Fatal(err)
		}
		secrets, err := getSecret(stageSpec.DelegationTokensSecret.SecretID, stageSpec.DelegationTokensSecret.SecretVersionID)
		if err != nil {
			log.Fatal(err)
		}
		err = tmpl.Execute(os.Stdout, struct {
			GetSecretValue func(string) string
		}{
			GetSecretValue: func(key string) string {
				value, ok := secrets[key]
				if !ok {
					log.Fatalf("Key %s was not found in secrets", key)
				}
				b, err := json.Marshal(value)
				if err != nil {
					log.Fatal(err)
				}
				return string(b)
			},
		})
		if err != nil {
			log.Fatal(err)
		}
		return nil
	},
}

func init() {
	cmd := deployTemplateCmd
	cmd.Flags().String("stage-id", "", "Deploy stage id")
	rootCmd.AddCommand(cmd)
}
