package ctl

import (
	"context"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"path"
	"strings"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/file"
	"github.com/vbauerster/mpb"
	"github.com/vbauerster/mpb/decor"
)

type Cli struct {
	Ctx           context.Context
	Config        Config
	OAuthToken    string
	QloudAPIURL   string
	SolomonAPIURL string
	YcAPIURL      string
	YcIAMAPIURL   string
}

func (cli *Cli) Initialize(ctx context.Context, configPath string) error {
	cli.Ctx = ctx

	if configPath != "" {
		loader := confita.NewLoader(file.NewBackend(configPath))
		err := loader.Load(context.Background(), &cli.Config)
		if err != nil {
			log.Fatal(err)
		}
	}

	home, _ := os.UserHomeDir()
	tokenPath := path.Join(home, ".dutytool", "token-oauth")
	token, err := ioutil.ReadFile(tokenPath)
	tokenStr := strings.Trim(string(token), "\n")

	if err == nil {
		cli.OAuthToken = tokenStr
	} else {
		fmt.Fprintf(os.Stderr, "File %s not found", tokenPath)
	}

	return nil
}

func (cli *Cli) CreateProgressBar(amountEnvs int, prefix string) (*mpb.Progress, *mpb.Bar) {
	progressBar := mpb.New(mpb.WithWidth(64), mpb.WithOutput(os.Stderr))
	return progressBar, progressBar.AddBar(int64(amountEnvs),
		mpb.PrependDecorators(
			decor.Name(prefix),
			decor.CountersNoUnit("%d / %d", decor.WCSyncWidth),
		),
		mpb.AppendDecorators(decor.Percentage()),
	)
}
