package interactions

import (
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"
)

type baseClient struct {
	baseURL string
	name    string
	client  http.Client
}

func (c *baseClient) makeRequest(requestURL string, method string, headers map[string]string, params url.Values, body interface{}) (interface{}, error) {
	var bodyReader io.Reader
	if body != nil {
		bodyReader, _ = body.(io.Reader)
	}

	req, err := http.NewRequest(method, c.baseURL+requestURL, bodyReader)
	if err != nil {
		return nil, err
	}

	for key, value := range headers {
		req.Header.Add(key, value)
	}

	// see: https://stackoverflow.com/a/56985985/16415965
	q := req.URL.Query()

	for k := range params {
		q.Add(k, params.Get(k))
	}

	req.URL.RawQuery = q.Encode()

	switch v := body.(type) {
	case url.Values:
		for k := range v {
			req.Form.Add(k, v.Get(k))
		}
	}

	resp, _ := c.client.Do(req)
	return c.processResponse(resp)
}

func (c *baseClient) processResponse(resp *http.Response) (interface{}, error) {
	if resp.StatusCode >= 400 {
		defer func() { _ = resp.Body.Close() }()
		body, _ := ioutil.ReadAll(resp.Body)
		return nil, fmt.Errorf("name=%s status=%d response=%s", c.name, resp.StatusCode, string(body))
	}

	return json.NewDecoder(resp.Body), nil
}

func (c *baseClient) defaultHeaders() map[string]string {
	return map[string]string{}
}

func (c *baseClient) get(url string, headers map[string]string, params url.Values) (interface{}, error) {
	return c.makeRequest(url, "GET", headers, params, nil)
}

func (c *baseClient) post(url string, headers map[string]string, params url.Values, data interface{}) (interface{}, error) {
	return c.makeRequest(url, "POST", headers, params, data)
}
