package interactions

import (
	"context"
	"fmt"
	"log"

	"a.yandex-team.ru/mail/swat/dutytool/pkg/ctl"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yp/go/yp"
)

type YPDataCenter string

const (
	YPDataCenterXDC YPDataCenter = "xdc"
	YPDataCenterSAS YPDataCenter = "sas"
	YPDataCenterVLA YPDataCenter = "vla"
	YPDataCenterMAN YPDataCenter = "man"
	YPDataCenterIVA YPDataCenter = "iva"
	YPDataCenterMYT YPDataCenter = "myt"
)

// Объекты Деплоя находятся в базе данных YP. Поэтому, на самом деле, это YP API client.
// Эта схема поможет лучше понять иерархию сущностей https://wiki.yandex-team.ru/deploy/docs/concepts/entity-hierarchy/
type DeployClient struct {
	clients map[YPDataCenter]*yp.Client
}

type DeployResources struct {
	CPU     uint64
	Mem     uint64
	Disk    uint64
	DiskBW  uint64
	Network uint64
}

type DeployReplicaSet struct {
	ID          string
	DataCenters []YPDataCenter
}

type DeployUnit struct {
	ID          string
	StageID     string
	ReplicaSets []DeployReplicaSet
	Resourses   DeployResources
}

type DeployStage struct {
	DeployUnits []DeployUnit
}

// Эта структура не является адекватной. Но пока что её хватает.
// На самом деле у POD'а есть не IP, а подсеть IP6 /112
// На каком-то из адресов висит сам POD. Но обычно нас интересует не сам POD, а контейнер, в котором бежит приложение.
// У него другой IP адрес. Именно этот адрес я кладу в эту структуру.
// А вот Host ссылается на контейнер POD'а. У контейнера приложения, кстати, вообще нету DNS имени.
type DeployPod struct {
	ID   string
	Host string
	IP   string
}

func (u *DeployUnit) ReplicaSetsInDC(dc YPDataCenter) []DeployReplicaSet {
	replicaSets := make([]DeployReplicaSet, 0)
	for _, replicaSet := range u.ReplicaSets {
		hasMatch := false
		for _, replicaSetDC := range replicaSet.DataCenters {
			if replicaSetDC == dc {
				hasMatch = true
				break
			}
		}

		if hasMatch {
			replicaSets = append(replicaSets, replicaSet)
		}
	}

	return replicaSets
}

func CreateDeployClient(cli ctl.Cli) DeployClient {
	if cli.OAuthToken == "" {
		log.Fatal("OAuthToken is not set, exit")
	}

	clients := make(map[YPDataCenter]*yp.Client)
	for _, dc := range []YPDataCenter{YPDataCenterXDC, YPDataCenterSAS, YPDataCenterVLA, YPDataCenterIVA, YPDataCenterMYT, YPDataCenterMAN} {
		client, err := yp.NewClient(string(dc), yp.WithAuthToken(cli.OAuthToken))

		if err != nil {
			log.Fatalf("Unable to create YP client (Ya.Deploy) to %s dc, continue...\n", dc)
		} else {
			clients[dc] = client
		}
	}

	return DeployClient{
		clients: clients,
	}
}

func (s *DeployClient) GetStage(stageID string) (deployStage DeployStage, err error) {
	response, err := s.clients[YPDataCenterXDC].GetObject(context.Background(), yp.GetObjectRequest{
		ObjectType: yp.ObjectTypeStage,
		ObjectID:   stageID,
		Format:     yp.PayloadFormatProto,
		Selectors:  []string{"/status", "/spec"},
	})

	if err != nil {
		return
	}

	// При анализе stage мы можем опираться либо на Spec (=как должно быть?), либо на Status (=как есть на самом деле?)
	// Я выбрал Status. Потому что нам в любом случае пришлось бы лезть в Status, чтобы узнать ReplicaSetID
	stage := &ypapi.TStage{
		Status: &ypapi.TStageStatus{}, // https://a.yandex-team.ru/arc/trunk/arcadia/yp/client/api/proto/stage.proto?rev=r7449161#L86
		Spec:   &ypapi.TStageSpec{},   // https://a.yandex-team.ru/arc/trunk/arcadia/yp/client/api/proto/stage.proto?rev=r7449161#L41
	}
	err = response.Fill(stage.Status, stage.Spec)
	if err != nil {
		return
	}

	deployUnits := make([]DeployUnit, 0)
	for deployUnitID, deployUnitStatus := range stage.Status.DeployUnits {
		deployUnit := DeployUnit{
			ID:      deployUnitID,
			StageID: stageID,
		}

		for deployUnitIDSpec, deployUnitSpec := range stage.Spec.DeployUnits {
			if deployUnitID == deployUnitIDSpec {
				if rs := deployUnitSpec.GetReplicaSet(); rs != nil {
					deployUnit.Resourses.CPU = *rs.ReplicaSetTemplate.PodTemplateSpec.Spec.ResourceRequests.VcpuLimit
					deployUnit.Resourses.Mem = *rs.ReplicaSetTemplate.PodTemplateSpec.Spec.ResourceRequests.MemoryLimit
					if networkBandwidthLimit := rs.ReplicaSetTemplate.PodTemplateSpec.Spec.ResourceRequests.NetworkBandwidthLimit; networkBandwidthLimit != nil {
						deployUnit.Resourses.Network = *networkBandwidthLimit
					}
					for _, request := range rs.ReplicaSetTemplate.PodTemplateSpec.Spec.DiskVolumeRequests {
						deployUnit.Resourses.Disk += request.GetQuotaPolicy().GetCapacity()
						deployUnit.Resourses.DiskBW += request.GetQuotaPolicy().GetBandwidthLimit()
					}
				} else if rs := deployUnitSpec.GetMultiClusterReplicaSet(); rs != nil {
					if cpu := rs.ReplicaSet.PodTemplateSpec.Spec.ResourceRequests.VcpuLimit; cpu != nil {
						deployUnit.Resourses.CPU = *cpu
					}

					if memoryLimit := rs.ReplicaSet.PodTemplateSpec.Spec.ResourceRequests.MemoryLimit; memoryLimit != nil {
						deployUnit.Resourses.Mem = *memoryLimit
					}
					if network := rs.ReplicaSet.PodTemplateSpec.Spec.ResourceRequests.NetworkBandwidthLimit; network != nil {
						deployUnit.Resourses.Network = *network
					}

					for _, request := range rs.ReplicaSet.PodTemplateSpec.Spec.DiskVolumeRequests {
						deployUnit.Resourses.Disk += request.GetQuotaPolicy().GetCapacity()
						deployUnit.Resourses.DiskBW += request.GetQuotaPolicy().GetBandwidthLimit()
					}
				}
			}
		}

		if rs := deployUnitStatus.GetReplicaSet(); rs != nil {
			for clusterID, clusterStatus := range rs.ClusterStatuses {
				deployUnit.ReplicaSets = append(deployUnit.ReplicaSets, DeployReplicaSet{
					ID:          clusterStatus.ReplicaSetId,
					DataCenters: []YPDataCenter{YPDataCenter(clusterID)},
				})
			}
		} else if rs := deployUnitStatus.GetMultiClusterReplicaSet(); rs != nil {
			replicaSet := DeployReplicaSet{
				ID: rs.ReplicaSetId,
			}
			replicaSet.DataCenters = make([]YPDataCenter, 0)
			for clusterID := range rs.ClusterStatuses {
				replicaSet.DataCenters = append(replicaSet.DataCenters, YPDataCenter(clusterID))
			}

			deployUnit.ReplicaSets = append(deployUnit.ReplicaSets, replicaSet)
		}

		deployUnits = append(deployUnits, deployUnit)
	}

	deployStage = DeployStage{
		DeployUnits: deployUnits,
	}
	err = nil
	return
}

func (s *DeployClient) GetPods(podSetID string, dc YPDataCenter) ([]DeployPod, error) {
	deployPods := make([]DeployPod, 0)
	client, ok := s.clients[dc]
	if !ok {
		return nil, fmt.Errorf("YP client in DC %s absent", dc)
	}

	response, err := client.SelectObjects(context.Background(), yp.SelectObjectsRequest{
		ObjectType: yp.ObjectTypePod,
		Filter:     fmt.Sprintf(`[/meta/pod_set_id] = "%s"`, podSetID),
		Format:     yp.PayloadFormatProto,
		Selectors:  []string{"/meta", "/spec", "/status"},
		Limit:      1000,
	})
	if err != nil {
		return nil, err
	}

	for response.Next() {
		pod := &ypapi.TPod{
			Meta:   &ypapi.TPodMeta{},
			Spec:   &ypapi.TPodSpec{},
			Status: &ypapi.TPodStatus{}, // https://a.yandex-team.ru/arc/trunk/arcadia/yp/client/api/proto/data_model.proto?rev=7449161
		}
		err = response.Fill(pod.Meta, pod.Spec, pod.Status)
		if err != nil {
			return nil, err
		}

		deployPod := DeployPod{
			ID: podSetID,
		}

		agent := pod.Status.Agent
		if agent == nil {
			log.Fatalf("Agent is nil for pod %s\n", podSetID)
		}

		for _, box := range agent.PodAgentPayload.Status.Boxes {
			if box.SpecificType == "default" { // Системные box'ы имеют тип "system".
				deployPod.IP = box.Ip6Address
			}
		}
		deployPod.Host = *pod.Status.Dns.PersistentFqdn
		deployPods = append(deployPods, deployPod)
	}
	return deployPods, nil
}

func (s *DeployClient) Close() {
	for _, client := range s.clients {
		client.Close()
	}
}

func (u *DeployUnit) Fullname() string {
	return fmt.Sprintf("%s.%s", u.StageID, u.ID)
}
