package interactions

import (
	"encoding/json"
	"fmt"
	"log"
	"strings"

	"a.yandex-team.ru/mail/swat/dutytool/pkg/ctl"
)

type QloudInstance struct {
	Host       string `json:"host"`
	Line       string `json:"line"`
	InstanceIP string `json:"instanceIp"`
	InstanceID string `json:"instanceId"`
}

type QloudComponent struct {
	ProjectName     string `json:"projectName"`
	ApplicationName string `json:"applicationName"`
	EnvironmentName string `json:"environmentName"`
	Name            string `json:"name"`
	Repository      string `json:"repository"`
	Event           struct {
		ComponentID string `json:"componentId"`
	} `json:"event"`
	RunningInstances []QloudInstance `json:"runningInstances"`
}

func (c QloudComponent) Fullname(format string) string {
	switch format {

	case "conductor":
		return fmt.Sprintf("%s_%s_%s_%s", c.ProjectName, c.ApplicationName, c.EnvironmentName, c.Name)
	default:
		return fmt.Sprintf("%s.%s.%s.%s", c.ProjectName, c.ApplicationName, c.EnvironmentName, c.Name)
	}
}

func (c QloudComponent) InstancesInDC(dc string) []QloudInstance {
	result := make([]QloudInstance, 0)
	for _, instance := range c.RunningInstances {
		if strings.HasPrefix(strings.ToLower(instance.Line), dc) {
			result = append(result, instance)
		}
	}
	return result
}

type QloudEnv struct {
	Components map[string]QloudComponent `json:"components"`
}

type QloudClient struct {
	baseClient
	token string
}

func CreateQloudClient(cli ctl.Cli) QloudClient {
	if cli.OAuthToken == "" {
		log.Fatal("OAuthToken is not set, exit")
	}

	return QloudClient{
		baseClient: baseClient{
			baseURL: cli.QloudAPIURL,
			name:    "qloud",
		},
		token: cli.OAuthToken,
	}
}

func (c *QloudClient) defaultHeaders() map[string]string {
	return map[string]string{
		"Authorization": "OAuth " + c.token,
	}
}

func (c *QloudClient) GetEnvironment(project string, application string, environment string) (QloudEnv, error) {
	path := fmt.Sprintf("/api/v1/environment/stable/%s.%s.%s", project, application, environment)

	var qloudEnv QloudEnv
	decoder, err := c.get(path, c.defaultHeaders(), nil)
	if err == nil {
		err = decoder.(*json.Decoder).Decode(&qloudEnv)
	}
	return qloudEnv, err
}
