package interactions

import (
	"bytes"
	"encoding/json"
	"fmt"
	"log"
	"sort"
	"time"

	"a.yandex-team.ru/mail/swat/dutytool/pkg/ctl"
)

type SolomonResponse struct {
	Vector []struct {
		Timeseries struct {
			Labels struct {
				Signal     string `json:"signal"`
				Percentile string `json:"percentile"`
			} `json:"labels"`
			Timestamps []int     `json:"timestamps"`
			Values     []float64 `json:"values"`
		} `json:"timeseries"`
	} `json:"vector"`
}

func (resp *SolomonResponse) avg(values []float64) float64 {
	avg := 0.0
	for _, val := range values {
		avg += val
	}
	return avg / float64(len(values))
}

func (resp *SolomonResponse) Avg() float64 {
	return resp.avg(resp.Vector[0].Timeseries.Values)
}

func (resp *SolomonResponse) Median() float64 {
	values := resp.Vector[0].Timeseries.Values
	sort.Slice(values, func(i, j int) bool {
		return values[i] < values[j]
	})

	if len(values) > 0 {
		return values[len(values)/2]
	} else {
		return 0
	}
}

func (resp *SolomonResponse) Max(smoth int) float64 {
	values := resp.Vector[0].Timeseries.Values
	sort.Slice(values, func(i, j int) bool {
		return values[i] < values[j]
	})

	if len(values) > 0 {
		return resp.avg(values[len(values)-smoth-1:])
	} else {
		return 0
	}
}

type SolomonClient struct {
	baseClient
	token string
}

func CreateSolomonClient(cli ctl.Cli) SolomonClient {
	if cli.OAuthToken == "" {
		log.Fatal("OAuthToken is not set, exit")
	}

	return SolomonClient{
		baseClient: baseClient{
			baseURL: cli.SolomonAPIURL,
			name:    "solomon",
		},
		token: cli.OAuthToken,
	}
}

func (c *SolomonClient) defaultHeaders() map[string]string {
	return map[string]string{
		"Content-Type":  "application/json",
		"Authorization": "OAuth " + c.token,
	}
}

func (c *SolomonClient) GetMetrics(project string, from time.Time, to time.Time, program string) (*SolomonResponse, error) {
	type Body struct {
		Downsampling struct {
			Aggregation         string `json:"aggregation"`
			Disabled            bool   `json:"disabled"`
			Fill                string `json:"fill"`
			GridMillis          uint   `json:"gridMillis"`
			IgnoreMinStepMillis bool   `json:"ignoreMinStepMillis"`
			MaxPoints           uint   `json:"maxPoints"`
		} `json:"downsampling"`
		From    int64  `json:"from"`
		To      int64  `json:"to"`
		Program string `json:"program"`
	}

	body := Body{
		From:    from.UnixMilli(),
		To:      to.UnixMilli(),
		Program: program,
	}
	body.Downsampling.Aggregation = "MAX"
	body.Downsampling.Disabled = true
	body.Downsampling.Fill = "PREVIOUS"
	body.Downsampling.GridMillis = 3600000
	body.Downsampling.IgnoreMinStepMillis = true
	body.Downsampling.MaxPoints = 500

	out, err := json.Marshal(body)
	if err != nil {
		return nil, err
	}

	decoder, err := c.post(
		fmt.Sprintf("/api/v2/projects/%s/sensors/data", project),
		c.defaultHeaders(),
		nil,
		bytes.NewBuffer(out),
	)
	if err != nil {
		return nil, err
	}

	response := &SolomonResponse{}
	err = decoder.(*json.Decoder).Decode(response)
	if err != nil {
		return nil, err
	}

	return response, nil
}
