import React from 'react';
import {Link, Redirect} from 'react-router-dom';
import {actions} from "stores";
import map from 'lodash/map';
import find from 'lodash/find';
import PT from 'prop-types';
import get from 'lodash/get';
import Modal from "shared/components/Modal";
import CartItem from './components/CartItem';
import GooglePayButton from './components/GooglePayButton';
import YandexPayButton from './components/YandexPayButton';

import './Cart.css';
import {connect} from "react-redux";

const mapStateToProps = state => {
    return {
        loading: state.ui.loading,

        methods: state.lpm.methods,

        goods: state.goods.goods,
        currentGood: state.goods.currentGood,

        cartSuccess: state.cart.cartSuccess,
        cartShow: state.cart.cartShow,
        cartLoading: state.cart.cartLoading,
        cartError: state.cart.cartError,
        cart: state.cart.cart,
    };
};

const mapDispatchToProps = dispatch => ({
    actions: {
        loadStart: () => dispatch(actions.ui.loadStart()),
        addGoodToCart: (id) => dispatch(actions.cart.addGoodToCart(id)),
        deleteFromCart: (id) => dispatch(actions.cart.deleteFromCart(id)),
        setCart: (cart) => dispatch(actions.cart.setCart(cart)),
        setCartPromocode: (promocode) => dispatch(actions.cart.setCartPromocode(promocode)),
        setCartStatus: (data) => dispatch(actions.cart.setCartStatus(data)),

        startCartPaymentsViaSDK: (email, mode, payToken, method) => dispatch(actions.cart.startCartPaymentsViaSDK(email, mode, payToken, method)),
        startCartGooglePay: (email) => dispatch(actions.cart.startCartGooglePay(email)),
        startCartYandexPay: (data) => dispatch(actions.cart.startCartYandexPay(data))
    }
});

const AuthMethodMode = Object.freeze({
    PanOnly: 'pan-only',
    CloudToken: 'cloud-token',
    Both: 'both',
})

class Cart extends React.Component {
    constructor(props) {
        super(props);
        this.form = React.createRef();

        this.state = {
            email: '',
            promocode: props.cart.promocode || '',
            method: 'paymentsViaSDK',
            mode: 'prod',
            payToken: '',
            authMethodMode: AuthMethodMode.CloudToken,
            close: false,
        }

        this.handleClose = this.handleClose.bind(this);
        this.handleSubmitPromocode = this.handleSubmitPromocode.bind(this);
        this.handleMethod = this.handleMethod.bind(this);
        this.handleEmail = this.handleEmail.bind(this);
        this.handlePromocode = this.handlePromocode.bind(this);
        this.handleMode = this.handleMode.bind(this);
        this.handleAuthMethodMode = this.handleAuthMethodMode.bind(this);
        this.handlePayToken = this.handlePayToken.bind(this);
        this.handleSubmit = this.handleSubmit.bind(this);
    }

    get method() {
        const {method} = this.state;
        const {methods} = this.props;
        return find(methods, ({value}) => value === method)
    }

    handleClose() {
        const {actions} = this.props;
        actions.setCartStatus({
            cartError: null,
            cartSuccess: false,
            cartLoading: false,
        })
        this.setState({close: true});
    }

    handleSubmitPromocode(e) {
        e.preventDefault();
        const {promocode} = this.state;
        const {actions} = this.props;
        actions.setCartPromocode(promocode);
    }

    handleEmail(e) {
        this.setState({email: e.target.value})
    }

    handleMethod(e) {
        this.setState({method: e.target.value})
    }

    handlePromocode(e) {
        this.setState({promocode: e.target.value})
    }

    handleMode(e) {
        if (e.target.checked) {
            this.setState({mode: e.target.value})
        }
    }

    handleAuthMethodMode(e) {
        if (e.target.checked) {
            const values = {
                panOnly: AuthMethodMode.PanOnly,
                cloudToken: AuthMethodMode.CloudToken,
                both: AuthMethodMode.Both,
            }
            this.setState({authMethodMode: values[e.target.value]})
        }
    }

    handlePayToken(e) {
        this.setState({payToken: e.target.value})
    }

    handleSubmit(e) {
        if (e.preventDefault) {
            e.preventDefault();
        }

        const {loading} = this.props;
        if (loading) {
            return;
        }

        const {startCartPaymentsViaSDK, startCartGooglePay, startCartYandexPay} = this.props.actions;
        const {email, mode, payToken, method} = this.state;

        switch (method) {
            case 'paymentsViaSDK':
                startCartPaymentsViaSDK(email, mode, payToken, method);
                break
            case 'googlePay':
                startCartGooglePay(email);
                break
            case 'yandexPay':
                startCartYandexPay({...e});
                break
        }
    }

    componentDidMount() {
        try {
            const {searchParams} = new URL(location.href);

            const invoiceID = searchParams.get('yandexPayInvoiceID');
            const skipEvents = (searchParams.get('skipEvents') || '').split(',');

            if (invoiceID) {
                window.history && history.replaceState({}, document.title, location.pathname + location.hash);

                const {startCartYandexPay} = this.props.actions;
                startCartYandexPay({invoiceID, skipEvents});
            }
        } catch (e) {
            console.error(e);
        }
    }

    renderPayButton() {
        const {cart, cartLoading} = this.props;
        const {email, authMethodMode} = this.state;
        const {method} = this;

        const goods = get(cart, 'goods', [])
        const total = goods.map(good => good.amount * good.price).reduce((a, b) => a + b, 0);

        switch (method && method.value) {
            case 'applePay':
                return (
                    <div className="cart__button" key="submit">
                        <button lang="ru"
                                type="submit"
                                disabled={cartLoading}
                                className="button button_full cart__button-apple"
                                style={{
                                    WebkitAppearance: '-apple-pay-button',
                                    ApplePayButtonType: 'plain',
                                    ApplePayButtonStyle: 'black',
                                }}
                        />
                    </div>
                )
            case 'yandexPay':
                const authModeToAuthMethods = {
                    [AuthMethodMode.CloudToken]: [YaPay.AllowedAuthMethod.CloudToken],
                    [AuthMethodMode.PanOnly]: [YaPay.AllowedAuthMethod.PanOnly],
                    [AuthMethodMode.Both]: [YaPay.AllowedAuthMethod.CloudToken, YaPay.AllowedAuthMethod.PanOnly],
                }
                const allowedAuthMethods = authModeToAuthMethods[authMethodMode]

                const paymentData = {
                    env: YaPay.PaymentEnv.Production,
                    version: 2,
                    countryCode: YaPay.CountryCode.Ru,
                    currencyCode: YaPay.CurrencyCode.Rub,
                    merchant: {
                        id: get(method, 'params.merchantId'),
                        name: get(method, 'params.merchantName')
                    },
                    order: {
                        id: cart.hash,
                        total: {amount: total},
                        items: goods.map(good => ({
                            label: good.title,
                            amount: good.amount * good.price
                        }))
                    },
                    paymentMethods: [
                        {
                            type: YaPay.PaymentMethodType.Card,
                            gateway: get(method, 'params.gateway'),
                            gatewayMerchantId: get(method, 'params.gatewayMerchantId'),
                            allowedAuthMethods: allowedAuthMethods,
                            allowedCardNetworks: [
                                YaPay.AllowedCardNetwork.Visa,
                                YaPay.AllowedCardNetwork.Mastercard,
                                YaPay.AllowedCardNetwork.Mir,
                                YaPay.AllowedCardNetwork.Uzcard
                            ]
                        }
                    ],
                    requiredFields: {
                        billingContact: {email: true}
                    }
                };

                return (
                    <div className="cart__button cart__button_yapay" key="submit">
                        <YandexPayButton
                            onEvent={this.handleSubmit}
                            paymentData={paymentData}
                            disabled={!Boolean(email)}
                        />
                    </div>
                );
            case 'googlePay':
                return (
                    <div className="cart__button" key="submit">
                        <GooglePayButton
                            method={method}
                            onClick={e => this.handleSubmit(e)}
                        />
                    </div>
                );
            default:
                let submitText;

                if (cartLoading) {
                    submitText = 'Оплачиваем...';
                } else {
                    submitText = `Оплатить ${goods.map(good => good.amount * good.price).reduce((a, b) => a + b, 0)} ₽`;
                }

                return (
                    <div className="cart__button" key="submit">
                        <button type="submit" disabled={cartLoading} className="button button_full button_action">
                            {submitText}
                        </button>
                    </div>
                )
        }
    }

    renderForm() {
        const {cart, cartError, cartSuccess, cartLoading, methods, actions: {deleteFromCart}} = this.props;
        const {email, mode, method, authMethodMode, promocode, payToken} = this.state;

        const goods = get(cart, 'goods', [])

        if (goods.length === 0 && !cartSuccess) {
            return (
                <div className="cart__empty">
                    <span className="cart__empty-pic">🤷‍♀</span>
                    Корзина пуста
                </div>
            )
        }

        if (cartSuccess) {
            return (
                <div className="cart__empty">
                    <span className="cart__empty-pic">✌️</span>
                    Заказ оплачен
                </div>
            )
        }

        return (
            <form ref={this.form} onSubmit={method === 'paymentsViaSDK' ? this.handleSubmit : e => e.preventDefault()}>
                <div className="cart__items">
                    {goods.map((item, i) => (
                        <CartItem disabled={cartLoading} onDelete={deleteFromCart} key={i} {...item} />
                    ))}
                </div>

                {cartError && <div className="cart__error">{cartError}</div>}

                <div className="cart__email">
                    <label className="cart__label">Промокод:</label>
                    <div className="promocode">
                        <input
                            onChange={this.handlePromocode}
                            disabled={cartLoading || cart.promocode}
                            name="promocode"
                            type="text"
                            className="input input_button-right input_full"
                            value={promocode}
                        />
                        <button
                            disabled={cartLoading || cart.promocode}
                            onClick={this.handleSubmitPromocode}
                            className="button button_input-left"
                        >
                            Применить
                        </button>
                    </div>
                </div>

                {method !== 'yandexPay' && (
                    <div className="cart__email">
                        <label className="cart__label">Электропочта:</label>
                        <input
                            onChange={this.handleEmail}
                            disabled={cartLoading}
                            name="email"
                            type="email"
                            className="input input_full"
                            required
                            value={email}
                        />
                    </div>
                )}

                <div className="cart__email">
                    <label className="cart__label">Способ оплаты:</label>
                    <div>
                        {map(methods, ({value, label}, i) => (
                            <label className="cart__mode-radio-label" key={i}>
                                <input required
                                       className="input input_radio "
                                       onChange={this.handleMethod}
                                       name="method"
                                       disabled={cartLoading}
                                       value={value}
                                       checked={value === method}
                                       type="radio"
                                />
                                {label}
                            </label>
                        ))}
                    </div>
                </div>

                {method === 'yandexPay' && (
                    <>
                        <div className="cart__mode" key="mode">
                            <label className="cart__mode-label">Режим:</label>
                            <label className="cart__mode-radio-label">
                                <input
                                    disabled={cartLoading}
                                    onChange={this.handleAuthMethodMode}
                                    checked={authMethodMode === AuthMethodMode.CloudToken}
                                    name="authMethodMode"
                                    value="cloudToken"
                                    type="radio"
                                    className="input input_radio"
                                />
                                CloudToken
                            </label>
                            <label className="cart__mode-radio-label">
                                <input
                                    disabled={cartLoading}
                                    name="authMethodMode"
                                    value="panOnly"
                                    type="radio"
                                    className="input input_radio"
                                    onChange={this.handleAuthMethodMode}
                                    checked={authMethodMode === AuthMethodMode.PanOnly}
                                />
                                PanOnly
                            </label>
                            <label className="cart__mode-radio-label">
                                <input
                                    disabled={cartLoading}
                                    onChange={this.handleAuthMethodMode}
                                    checked={authMethodMode === AuthMethodMode.Both}
                                    name="authMethodMode"
                                    value="both"
                                    type="radio"
                                    className="input input_radio"
                                />
                                Both
                            </label>
                        </div>
                    </>
                )}

                {['paymentsViaSDK', 'payments'].find(item => item === method) && (
                    <>
                        <div className="cart__mode" key="mode">
                            <label className="cart__mode-label">Режим:</label>
                            <label className="cart__mode-radio-label">
                                <input
                                    disabled={cartLoading}
                                    onChange={this.handleMode}
                                    checked={mode === 'prod'}
                                    name="mode"
                                    value="prod"
                                    type="radio"
                                    className="input input_radio"
                                />
                                Продовый
                            </label>
                            <label className="cart__mode-radio-label">
                                <input
                                    disabled={cartLoading}
                                    name="mode"
                                    value="test"
                                    type="radio"
                                    className="input input_radio"
                                    onChange={this.handleMode}
                                    checked={mode === 'test'}
                                />
                                Тестовый
                            </label>
                            {method === 'paymentsViaSDK' && (
                                <label className="cart__mode-radio-label">
                                    <input
                                        disabled={cartLoading}
                                        name="mode"
                                        value="manual"
                                        type="radio"
                                        className="input input_radio"
                                        onChange={this.handleMode}
                                        checked={mode === 'manual'}
                                    />
                                    Свой pay_token
                                </label>
                            )}
                        </div>
                    </>
                )}

                {mode === 'manual' && (
                    <div className="cart__email" key="pay_token">
                        <label className="cart__email-label">pay_token:</label>
                        <input
                            onChange={this.handlePayToken}
                            disabled={cartLoading}
                            name="pay_token"
                            type="text"
                            className="input input_full"
                            required
                            value={payToken}
                        />
                    </div>
                )}

                {this.renderPayButton()}
            </form>
        )
    }

    render() {
        const {cart, cartLoading, actions} = this.props;
        const {close} = this.state;
        const goods = get(cart, 'goods', []);

        return (
            <Modal isOpen onRequestClose={this.handleClose}>
                {close && <Redirect to="/" />}

                <div className="cart">
                    {this.renderForm()}

                    <div className="cart__button">
                        <button
                            onClick={this.handleClose}
                            className="button button_full"
                            disabled={cartLoading}
                        >
                            {goods.length > 0 ? 'Отмена' : 'Закрыть'}
                        </button>
                    </div>
                    {cart.hash && (
                        <div className="cart__button">
                            <button
                                className="button button_danger button_full"
                                onClick={() => {
                                    actions.setCart({});
                                    this.handleClose();
                                }}
                                disabled={cartLoading}
                            >
                                Удалить корзину
                            </button>
                        </div>
                    )}
                </div>
            </Modal>
        )
    }

}

Cart.propTypes = {
    cart: PT.object,
    loading: PT.bool,
    cartSuccess: PT.bool,
    cartError: PT.string,
};

export default connect(mapStateToProps, mapDispatchToProps)(Cart);
