import {call, put, select, takeEvery} from 'redux-saga/effects';
import PaymentRequest from 'shared/lib/paymentRequest';
import api from 'shared/lib/api';
import {sleep, redirectForm} from "shared/utils";
import sum from 'lodash/sum';
import map from 'lodash/map';

import get from 'lodash/get';
import find from 'lodash/find';
import forEach from 'lodash/forEach';
import actions from './actions';
import lpmActions from '../lpm/actions';


function* addGoodToCart({id}) {
    yield put(actions.setCartStatus({cartError: null, cartLoading: true}));

    const state = yield select();

    const cart = {...state.cart.cart};
    cart.goods = cart.goods || [];

    const good = state.goods.goods.find(good => good.id === id);
    const goodInCart = cart.goods.find(good => good.id === id);

    if (Boolean(goodInCart)) {
        cart.goods = cart.goods.map(good => {
            if (good.id === id) {
                good.amount += 1;
            }
            return good;
        });
    } else {
        cart.goods.push({...good, amount: 1})
    }

    const {
        response,
        error
    } = yield call(api, '/api/cart' + (cart.hash ? `/${cart.hash}` : ''), 'POST', {goods: cart.goods});
    if (error || !response.data) {
        yield put(actions.setCartStatus({cartError: 'Ошибка добавления товара'}));
    } else {
        cart.goods = response.data.goods;
        cart.hash = response.data.hash;
        yield put(actions.setCart(cart));
    }
    yield put(actions.setCartStatus({cartLoading: false}));
}

function* setCartPromocode({promocode}) {
    yield put(actions.setCartStatus({cartError: null, cartLoading: true}));

    const {cart: state} = yield select();

    const cart = {...state.cart};

    const {
        response,
        error
    } = yield call(api, `/api/cart/${cart.hash}`, 'POST', {promocode});
    if (error || !response.data) {
        yield put(actions.setCartStatus({cartError: 'Ошибка применения промокода'}));
    } else {
        cart.promocode = response.data.promocode;
        yield put(actions.setCart(cart));
        yield put(lpmActions.loadLpm());
    }
    yield put(actions.setCartStatus({cartLoading: false}));
}

function* deleteGoodFromCart({id}) {
    yield put(actions.setCartStatus({cartError: null, cartLoading: true}));

    const {cart: state} = yield select();

    const cart = {...state.cart};
    cart.goods = (cart.goods || []).filter(item => item.id !== id);

    const {
        response,
        error
    } = yield call(api, '/api/cart' + (cart.hash ? `/${cart.hash}` : ''), 'POST', {goods: cart.goods});
    if (error || !response.data) {
        yield put(actions.setCartStatus({cartError: 'Ошибка удаления товара'}));
    } else {
        cart.goods = response.data.goods;
        cart.hash = response.data.hash;
        yield put(actions.setCart(cart));
    }
    yield put(actions.setCartStatus({cartLoading: false}));
}

function* startCartPaymentsViaSDK({email, mode, payToken}) {
    const {cart: state} = yield select();

    let userEmail = email;
    let paymentToken = payToken;
    let label, value, id;

    if (mode === 'manual') {
        label = 'Оплата через ввод pay_token';
        id = new Date().getTime();
        value = sum(map(state.cart.goods, ({amount, price}) => amount * price));
    } else {
        yield put(actions.setCartStatus({cartError: null, cartLoading: true}));

        const {response, error} = yield call(api, `/api/cart/${state.cart.hash}/startpaymentsviasdk`, 'POST', {
            email,
            mode
        });

        if (error || !response.data) {
            yield put(actions.setCartStatus({cartError: 'Ошибка создания корзины', cartLoading: false}));
            return;
        }

        userEmail = response.data.email;
        paymentToken = response.data.payToken;
        label = response.data.label;
        value = response.data.total;
        id = response.data.id;
    }

    const paymentRequest = new PaymentRequest(
        [{
            supportedMethods: 'yandex',
            data: {userEmail, paymentToken}
        }],
        {
            id: id,
            total: {label, amount: {currency: 'RUB', value}}
        }
    );

    try {
        yield call(paymentRequest.show.bind(paymentRequest));
        yield put(actions.setCart({}));
        yield put(actions.setCartStatus({
            cartError: null,
            cartSuccess: true,
            cartLoading: false,
        }));
    } catch (exception) {
        yield put(actions.setCartStatus({
            cartError: `Ошибка оплаты (PaymentRequest ${exception.message})`,
            cartLoading: false
        }));
    }
}

function* startCartGooglePay() {
    yield put(actions.setCartStatus({cartError: null, cartLoading: true}));

    const {lpm, cart} = yield select();
    const googlePayMethod = find(lpm.methods, ({value}) => value === 'googlePay');
    const value = sum(map(cart.cart.goods, ({amount, price}) => amount * price));

    const {paymentsClient, baseCardPaymentMethod} = googlePayMethod;

    const tokenizationSpecification = {
        type: 'PAYMENT_GATEWAY',
        parameters: {
            'gateway': get(googlePayMethod, 'params.gateway'),
            'gatewayMerchantId': get(googlePayMethod, 'params.gatewayMerchantId')
        }
    };


    const cardPaymentMethod = Object.assign(
        {},
        baseCardPaymentMethod,
        {
            tokenizationSpecification: tokenizationSpecification
        }
    );

    const paymentDataRequest = {
        apiVersion: 2,
        apiVersionMinor: 0
    };

    paymentDataRequest.allowedPaymentMethods = [cardPaymentMethod];

    paymentDataRequest.transactionInfo = {
        totalPriceStatus: 'FINAL',
        totalPrice: `${value}`,
        currencyCode: 'RUB',
        countryCode: 'RU'
    };

    paymentDataRequest.merchantInfo = {
        merchantId: get(googlePayMethod, 'params.merchantId'),
        merchantName: get(googlePayMethod, 'params.merchantName')
    };

    let token;
    try {
        const {paymentMethodData: {tokenizationData}} = yield call(() => paymentsClient.loadPaymentData(paymentDataRequest));
        token = tokenizationData.token;
    } catch (exception) {
        yield put(actions.setCartStatus({
            cartError: `Ошибка оплаты (Google Pay ${exception.message})`,
            cartLoading: false
        }));
        return;
    }

    const {response, error} = yield call(api, `/api/cart/${cart.cart.hash}/startgooglepay`, 'POST', {token});
    if (error || !response.data) {
        yield put(actions.setCartStatus({cartError: 'Ошибка проведения оплаты', cartLoading: false}));
        return;
    }

    yield put(actions.setCart({}));
    yield put(actions.setCartStatus({
        cartError: null,
        cartSuccess: true,
        cartLoading: false,
    }));
}

function* startCartYandexPay({data}) {
    const {token, error, email, paymentMethodInfo} = data;

    let {invoiceID, skipEvents, payment} = data;
    payment = window.YaPay && payment;

    if (error) {
        yield put(actions.setCartStatus({cartError: JSON.stringify(error)}));
        return;
    } else {
        yield put(actions.setCartStatus({cartError: null, cartLoading: true}));
    }

    const {cart} = yield select();

    if (!invoiceID) {
        const {
            response,
            error: errorStart
        } = yield call(api, `/api/cart/${cart.cart.hash}/startyandexpay`, 'POST', {
            token,
            email,
            paymentMethodInfo,
            redirectUrl: `${location.protocol}//${location.host}`,
        });

        if (errorStart || !response.data) {
            yield put(actions.setCartStatus({cartError: 'Ошибка проведения оплаты', cartLoading: false}));
            payment.complete('error');
            return;
        }
        invoiceID = response.data.invoiceID;
    }

    const events = {};
    forEach(skipEvents || [], id => {
        events[id] = true
    });

    let paymentSuccess = null;
    for (let i = 0; i <= 100; i += 1) {
        const {
            response: responseStatus,
            error: errorStatus
        } = yield call(api, `/api/cart/${cart.cart.hash}/getstatusyandexpay`, 'GET', {invoiceID});

        if (errorStatus) {
            yield put(actions.setCartStatus({
                cartError: 'Ошибка получения статуса платежа',
                cartLoading: false,
            }));
            payment && payment.complete('error');
            return;
        } else {
            forEach(responseStatus.data, ({id, changes}) => {
                id = String(id);
                if (!events.hasOwnProperty(id)) {
                    events[id] = changes;

                    forEach(changes, ({changeType, status, userInteraction}) => {
                        switch (changeType) {
                            case "PaymentInteractionRequested":
                                if (get(userInteraction, 'interactionType') === 'Redirect') {
                                    const params = {};
                                    forEach(get(userInteraction, 'request.form'), ({key, template}) => {
                                        const skipEvents = encodeURIComponent(Object.keys(events).join(','));
                                        const terminationUri = `${location.protocol}//${location.host}?skipEvents=${skipEvents}&yandexPayInvoiceID=${encodeURIComponent(invoiceID)}#/cart`

                                        params[key] = String(template)
                                            .replace(
                                                '{?termination_uri}',
                                                '?termination_uri=' + encodeURIComponent(terminationUri)
                                            ).replace(
                                                '{termination_uri}',
                                                terminationUri
                                            );
                                    });

                                    payment && payment.complete(YaPay.CompleteReason.AuthRedirect);

                                    switch (get(userInteraction, 'request.requestType')) {
                                        case 'BrowserPostRequest':
                                            redirectForm(get(userInteraction, 'request.uriTemplate'), params, 'post');
                                            break;
                                        case 'BrowserGetRequest':
                                            redirectForm(get(userInteraction, 'request.uriTemplate'), params, 'get');
                                            break;
                                    }
                                }
                                break
                            case "InvoiceStatusChanged":
                                if (status === 'paid') {
                                    paymentSuccess = true;
                                }
                                break
                            case 'PaymentStatusChanged':
                                if (status === 'failed') {
                                    paymentSuccess = false;
                                }
                                break
                        }
                    })
                }
            });
        }

        if (paymentSuccess === null) {
            yield call(sleep, 2000);
        } else {
            break;
        }
    }

    if (paymentSuccess === true) {
        yield put(actions.setCart({}));
        yield put(actions.setCartStatus({
            cartError: null,
            cartSuccess: true,
            cartLoading: false,
        }));
        payment && payment.complete(YaPay.CompleteReason.Success);
    } else if (paymentSuccess === false) {
        yield put(actions.setCartStatus({cartError: 'Ошибка оплаты', cartLoading: false}));
        payment && payment.complete(YaPay.CompleteReason.Error);
    } else {
        yield put(actions.setCartStatus({cartError: 'Ошибка получения статуса оплаты', cartLoading: false}));
        payment && payment.complete(YaPay.CompleteReason.Error);
    }
}

export default [
    takeEvery(actions.ADD_GOOD_TO_CART, addGoodToCart),
    takeEvery(actions.DELETE_FROM_CART, deleteGoodFromCart),
    takeEvery(actions.SET_CART_PROMOCODE, setCartPromocode),

    takeEvery(actions.START_CART_PAYMENTS_VIA_SDK, startCartPaymentsViaSDK),
    takeEvery(actions.START_CART_GOOGLE_PAY, startCartGooglePay),
    takeEvery(actions.START_CART_YANDEX_PAY, startCartYandexPay),
];
