import {call, put, takeEvery, select} from 'redux-saga/effects';
import get from "lodash/get";
import filter from "lodash/filter";
import api from 'shared/lib/api';
import actions from './actions';

function* load() {
    const state = yield select()
    const hash = state.cart.cart.hash;

    const {response} = yield call(api, '/api/lpm' + (hash ? `/${hash}` : ''));

    if (response) {
        for (const item of response.data) {
            const src = get(item, 'external.src');

            if (src) {
                const promise = new Promise(((scriptSrc) => {
                    return (resolve, reject) => {
                        const script = document.createElement('script');
                        script.onload = resolve;
                        script.onerror = reject;
                        script.src = scriptSrc;
                        document.head.appendChild(script);
                    }
                })(src));

                yield call(() => promise)
            }
        }

        const methods = []

        for (const method of filter(response.data, item => !get(item, 'hide', false))) {
            switch (method.value) {
                case 'applePay':
                    try {
                        if (window.ApplePaySession && ApplePaySession.canMakePayments()) {
                            methods.push(method)
                        }
                    } catch (ignore) {
                    }

                    break
                case 'googlePay':
                    const baseRequest = {apiVersion: 2, apiVersionMinor: 0};
                    const allowedCardNetworks = get(method, 'params.allowedCardNetworks');
                    const allowedCardAuthMethods = get(method, 'params.allowedCardAuthMethods');

                    const baseCardPaymentMethod = {
                        type: 'CARD',
                        parameters: {
                            allowedAuthMethods: allowedCardAuthMethods,
                            allowedCardNetworks: allowedCardNetworks
                        }
                    };

                    const isReadyToPayRequest = Object.assign({}, baseRequest);
                    isReadyToPayRequest.allowedPaymentMethods = [baseCardPaymentMethod];

                    const paymentsClient = new google.payments.api.PaymentsClient({
                        environment: get(method, 'params.environment')
                    });

                    const promise = new Promise((resolve, reject) => {
                        paymentsClient.isReadyToPay(isReadyToPayRequest)
                            .then(resolve)
                            .catch(reject)
                    })

                    try {
                        const response = yield call(() => promise)
                        if (response.result) {
                            methods.push({
                                ...method,
                                paymentsClient,
                                baseCardPaymentMethod,
                            })
                        }
                    } catch (err) {
                    }

                    break
                default:
                    methods.push(method);
            }
        }

        yield put(actions.setLpm(methods));
    }
}

export default [
    takeEvery(actions.LOAD_LPM, load),
];
