'use strict';
import path from 'path';
import webpack from 'webpack';
import CopyWebpackPlugin from 'copy-webpack-plugin';
import MiniCssExtractPlugin from 'mini-css-extract-plugin';
import TerserPlugin from 'terser-webpack-plugin';

const isProd = process.env.NODE_ENV === 'production';
const src = path.join(__dirname, 'src');
const dst = path.join(__dirname, '..', 'server', 'public', 'static');
const modulesDirectories = [src, path.join(src, 'js'), 'node_modules'];

const extractCss = new MiniCssExtractPlugin({ filename: 'css/[name].css' });

export default {
    entry: {
        main: ['@babel/polyfill', './src/js/main.js']
    },
    cache: {
        type: 'memory',
    },
    devtool: isProd ? false : 'eval-source-map',
    output: {
        path: dst,
        filename: 'js/main.js',
        publicPath: '/static/',
    },
    resolve: {
        modules: modulesDirectories,
        extensions: ['.js', '.jsx', '.css'],
        alias: {
            stores: path.join(src, 'js', 'stores'),
            shared: path.join(src, 'js', 'shared'),
        }
    },
    module: {
        rules: [
            {
                test: /\.jsx?$/,
                exclude: /node_modules/,
                loader: 'babel-loader'
            },
            {
                test: /\.css/,
                use: [
                    { loader: MiniCssExtractPlugin.loader },
                    { loader: 'css-loader' },
                ],
            },
            {
                test: /\.(jpg|jpeg|gif|png|ico|svg)$/,
                type: 'asset/resource',
                generator: {
                    filename: 'img/[name][ext]'
                }
            },
        ]
    },
    plugins: [
        extractCss,
        new webpack.DefinePlugin({
            IS_PRODUCTION: isProd,
        }),
        new CopyWebpackPlugin({
            patterns: [
                {
                    from: path.join(src, 'img'),
                    to: path.join(dst, 'img')
                }
            ]
        }),
    ],
    optimization: {
        minimize: isProd,
        moduleIds: isProd ? 'hashed' : 'named',
        minimizer: [
            new TerserPlugin({
                parallel: true,
                terserOptions: {
                    output: {
                        comments: false,
                    },
                },
            }),
        ],
        splitChunks: {
            cacheGroups: {
                default: false,
                commons: {
                    test: /node_modules/,
                    filename: 'js/vendor.main.js',
                    enforce: true,
                    chunks: 'all',
                },
            },
        },
    },
    watchOptions: {
        ignored: /node_modules/,
    },
};
