import logging

import requests

from ci.tasklet.common.proto import service_pb2 as ci
from tasklet.services.yav.proto import yav_pb2

from mail.swat.qloud_release.impl.qloud_client import QloudClient
from mail.swat.qloud_release.proto import qloud_release_tasklet

URL_TEMPLATE = "https://platform.yandex-team.ru/projects/{}"


logger = logging.getLogger(__name__)


class QloudReleaseImpl(qloud_release_tasklet.QloudReleaseBase):
    def get_qloud_token(self):
        secret_id = self.input.config.qloud_token_yav_secret_id

        spec = yav_pb2.YavSecretSpec(uuid=secret_id, key='qloud.token')
        secret = self.ctx.yav.get_secret(spec)
        return secret.secret

    def get_docker_image_url(self):
        resource = next(r for r in self.input.sb_resources if r.type == 'YA_PACKAGE')
        version = resource.attributes['resource_version']
        return version

    def get_docker_image_hash(self, image_url):
        registry_url, tag = image_url.split(':', 1)
        url = 'https://dockinfo.yandex-team.ru/api/docker/resolve?registryUrl={registry_url}&tag={tag}'.format(
            registry_url=registry_url,
            tag=tag
        )

        resp = requests.get(url)
        if resp.ok:
            return resp.json()['hash']
        else:
            raise Exception("Failed to get hash for image '{}.{}' with code {}: {}".format(
                registry_url, tag, resp.status_code, resp.text))

    def update_components(self, image_url, image_hash, token):
        properties = {'repository': image_url,
                      'hash': image_hash}

        client = QloudClient(
            token=token,
            logger=logger,
        )

        sorted_by_env_id = {}

        for component in self.input.config.components:
            env_id, component_id = component.rsplit('.', 1)
            sorted_by_env_id.setdefault(env_id, []).append(component_id)

        for environment, components in sorted_by_env_id.iteritems():
            components_id = map(lambda c: c.rsplit('.', 1)[-1], components)

            params = []
            for component_id in components_id:
                params.append({
                    "componentName": component_id,
                    "properties": properties
                })

            version = client.get_environment_deployed_version(environment)
            client.fast_deploy_environment(environment_id=environment,
                                           version=version,
                                           params=params,
                                           comment=None)

        for env_id in sorted_by_env_id.keys():
            client.wait_for_environment_deploy(env_id)

    def run(self):
        progress_msg = ci.TaskletProgress()
        progress_msg.job_instance_id.CopyFrom(self.input.context.job_instance_id)
        progress_msg.id = "releaseToQLOUD"
        progress_msg.module = "QLOUD"
        progress_msg.url = self._make_url(self.input.config.components[0].rsplit('.', 1)[0])
        progress_msg.status = ci.TaskletProgress.Status.RUNNING
        progress_msg.progress = 0
        self.ctx.ci.UpdateProgress(progress_msg)

        logger.info('hhaahaha')
        qloud_token = self.get_qloud_token()
        image_url = self.get_docker_image_url()
        image_hash = self.get_docker_image_hash(image_url)
        self.update_components(image_url, image_hash, qloud_token)

        progress_msg.progress = 1
        progress_msg.status = ci.TaskletProgress.Status.SUCCESSFUL
        self.ctx.ci.UpdateProgress(progress_msg)

    def _make_url(self, env_id):
        return URL_TEMPLATE.format(env_id.replace('.', '/'))
