import time

import requests

from sandbox.common import utils

URL_TEMPLATE = "https://platform.yandex-team.ru/projects/{}"


def utf_encode(text):
    return text.encode("utf-8") if isinstance(text, unicode) else text


class EnvStatus(utils.Enum):
    class Group(utils.GroupEnum):
        IDLE = None
        COMPLETE = None
        IN_PROGRESS = None
        ABNORMAL_TERMINATION = None

    with Group.IDLE:
        TEMPLATE = None
        NEW = None

    with Group.COMPLETE:
        CANCELED = None
        REMOVED = None
        FAILED = None
        DEPLOYED = None

    with Group.IN_PROGRESS:
        ALLOCATING = None
        CONFIGURING = None
        PREPARING = None
        DEPLOYING = None
        PENDING_REMOVE = None
        REMOVING = None
        ACTIVATING = None


EnvStatus.Group.ABNORMAL_TERMINATION = (EnvStatus.FAILED, EnvStatus.CANCELED)


class QloudClient(requests.Session):
    BASE_URL = 'https://qloud-ext.yandex-team.ru/'
    _deploy_timeout = 1800

    def __init__(self, token, logger):
        super(QloudClient, self).__init__()
        self._token = token
        self._logger = logger

    def request(self, method, url, params=None, data=None, headers=None, *args, **kwargs):
        headers = headers or {}
        headers['Authorization'] = 'OAuth {}'.format(self._token)
        self._logger.info('qloud-client request method:{} url:{} params:{}'.format(method, url, params))
        return super(QloudClient, self).request(method, url, params, data, headers, *args, **kwargs)

    def endpoint_url(self, relative_url):
        return self.BASE_URL + relative_url

    def get_environment_deployed_version(self, environment_id):
        url = self.endpoint_url("api/v1/environment/stable/{}".format(environment_id))
        res = self.get(url)
        assert res.ok, {'status_code': res.status_code, 'text': utf_encode(res.text)}
        self._logger.info('Get env deployed version, code:{} text:{}'.format(res.status_code, utf_encode(res.text)))
        versions = filter(lambda v: v['status'] == "DEPLOYED", res.json()['versions'])
        if not versions:
            return None
        return str(versions[0]['version'])

    def deploy_component_image(self, image, image_hash, environment_id, component, version, comment=None):
        url = self.endpoint_url("api/v1/component/{}.{}/{}/deploy".format(environment_id, component, version))
        if comment:
            url += "?comment={}".format(comment)
        params = {'repository': image,
                  'hash': image_hash}
        res = self.post(url, json=params)
        assert res.ok, {'status_code': res.status_code, 'text': utf_encode(res.text)}
        self._logger.info(
            'deploy component image code:{} text:{}'.format(res.status_code, utf_encode(res.text)))

    def fast_deploy_environment(self, environment_id, version, params, comment=None):
        url = self.endpoint_url('api/v1/environment/fast-deploy/{environment_id}/{version}'.format(
            environment_id=environment_id,
            version=version
        ))
        if comment:
            url += "?comment={}".format(comment)
        res = self.post(url, json=params)
        assert res.ok, {'status_code': res.status_code, 'text': utf_encode(res.text)}
        self._logger.info(
            'deploy environment image code:{} text:{}'.format(res.status_code, utf_encode(res.text)))
        return res

    def wait_for_environment_deploy(self, environment_id):
        self._logger.info('Waiting for environment deploy')
        status, res = self.wait_environment_status(environment_id, expected_statuses=EnvStatus.Group.COMPLETE,
                                                   timeout=self._deploy_timeout)
        if status != EnvStatus.DEPLOYED:
            error_text = "Deployment failed with unexpected status '{}'".format(str(status))
            if status == EnvStatus.FAILED:
                error_text += ": {}".format(res['statusMessage'])
            if not res.ok:
                error_text += ": request failed, code: {}, message: {}".format(res.status_code, utf_encode(res.text))
            raise Exception(error_text)

    def wait_environment_status(self, environment_id, expected_statuses, timeout, sleep_time=5):
        start_time = time.time()
        status = None

        while not status or status not in expected_statuses:
            if time.time() - start_time >= timeout:
                raise Exception("Wait timeout exceeded")

            time.sleep(sleep_time)
            res = self.get_environment_status(environment_id)
            if res.ok:
                status = EnvStatus[res.json()['status']]
            else:
                break

        return status, res

    def get_environment_status(self, environment_id):
        url = self.endpoint_url('api/v1/environment/status/{environment_id}'.format(
            environment_id=environment_id,
        ))
        res = self.get(url)
        self._logger.info('Get env status, code: {} text: {}'.format(res.status_code, utf_encode(res.text)))
        return res
