import logging
import os

from datetime import datetime, timedelta, timezone

from aiohttp import ClientSession, web

OAUTH = os.environ['OAUTH']
ORIGIN = os.environ.get('ORIGIN', 'http://localhost:8080')
PORT = int(os.environ.get('HTTP_PORT', '80'))
LOG_LEVEL = os.environ.get('LOG_LEVEL', 'INFO')

logging.basicConfig(level=getattr(logging, LOG_LEVEL))


def render_time(dt) -> str:
    dt = dt.astimezone(timezone.utc)
    display_dt = dt.astimezone(timezone(timedelta(hours=3)))
    now = datetime.now(tz=timezone.utc)
    if now.day == dt.day or now - dt < timedelta(hours=12):
        return display_dt.time().strftime('%H:%M')
    return display_dt.strftime('%D %T')


async def get_badge_params(request):
    try:
        review_id = str(int(request.match_info['review_id']))
    except (ValueError, KeyError):
        return 'INVALID_REVIEW', 'UNKNOWN', 'grey'
    async with ClientSession() as session:
        async with session.get(f'https://a.yandex-team.ru/api/review/review-request/{review_id}',
                               headers={'Authorization': f'OAuth {OAUTH}'}) as response:
            review = await response.json()

    if 'errors' in review:
        return review_id, 'UNKNOWN_REVIEW', 'grey'

    logging.debug('review %s' % review)
    try:
        check_id = review['ciCheckStatus']['checkId']
    except KeyError:
        return review_id, 'NO_CHECKS', 'grey'

    async with ClientSession() as session:
        async with session.get('https://ci.yandex-team.ru//api/v1.0/checks/' + check_id) as response:
            check = await response.json()

    logging.debug('check %s', check)
    updated_at = datetime.fromisoformat(review['lastUpdated'].replace('Z', '+00:00'))
    try:
        status = check['progress']['status']
        check_status = check.get('progress').get('check_status')
    except KeyError:
        return review_id, 'PROGRESS_UNKNOWN', 'grey'

    display_status = status if status != 'FINISHED' else check_status
    display_status = f'{display_status}@{render_time(updated_at)}'

    color = 'green'
    if check_status == 'FAILED':
        color = 'red'
    if status != 'FINISHED':
        color = 'yellow'
    return review_id, display_status, color


async def badge(request):
    review_id, display_status, color = await get_badge_params(request)
    location = f'https://codereview.common-int.yandex-team.ru/badges/{review_id}-{display_status}-{color}.svg'
    raise web.HTTPFound(location=location)


async def embed(request):
    review_id = str(int(request.match_info['review_id']))
    review_link = 'https://a.yandex-team.ru/review/{review_id}/'.format(review_id=review_id)
    image_link = ORIGIN + '/review/' + review_id + '/'
    response = '[![Review {review_id}]({image_link})]({review_link})'
    response = response.format(review_id=review_id, review_link=review_link, image_link=image_link)
    return web.Response(text=response)


app = web.Application()
app.add_routes([web.get('/review/{review_id}/', badge)])
app.add_routes([web.get('/review/{review_id}/embed/', embed)])
web.run_app(app, host='::', port=PORT)
