package ru.yandex.mail.so.templatemaster;

import java.io.IOException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import ru.yandex.cache.async.AsyncCacheResultType;
import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.ExecutorServiceCloser;
import ru.yandex.concurrent.LifoWaitBlockingQueue;
import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.server.async.DelegatedHttpAsyncRequestHandler;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.mail.so.templatemaster.cache.StableTemplateCache;
import ru.yandex.mail.so.templatemaster.cache.UnstableTemplateCache;
import ru.yandex.mail.so.templatemaster.config.ImmutableTemplateMasterConfig;
import ru.yandex.mail.so.templatemaster.indexing.Indexer;
import ru.yandex.mail.so.templatemaster.indexing.SaveTemplateHandler;
import ru.yandex.mail.so.templatemaster.searching.LocalMatchHandler;
import ru.yandex.mail.so.templatemaster.searching.RouteHandler;
import ru.yandex.mail.so.templatemaster.storage.LuceneUnstableAsyncIndexer;
import ru.yandex.search.proxy.universal.UniversalSearchProxy;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.EnumStaterFactory;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.stater.ThreadPoolStater;

public class TemplateMaster
    extends UniversalSearchProxy<ImmutableTemplateMasterConfig>
{
    private final AsyncClient luceneIndexClient;
    private final AsyncClient luceneSearchClient;
    private final AsyncClient producerAsyncClient;
    private final UnstableTemplateCache unstableCache;
    private final StableTemplateCache stableCache;
    private final ThreadPoolExecutor threadPool;
    private final TimeFrameQueue<Long> resultFound;
    private final TimeFrameQueue<AsyncCacheResultType> unstableCacheHits;
    private final TimeFrameQueue<AsyncCacheResultType> stableCacheHits;

    public TemplateMaster(ImmutableTemplateMasterConfig config)
        throws IOException
    {
        super(config);
        producerAsyncClient =
            client("AsyncProducer", config.producerAsyncClientConfig());
        luceneIndexClient =
            client("LuceneIndex", config.luceneIndexConfig());
        luceneSearchClient =
            client("LuceneSearch", config.luceneSearchConfig());

        LuceneUnstableAsyncIndexer persistentStorage =
            new LuceneUnstableAsyncIndexer(this);
        unstableCache = new UnstableTemplateCache(
            this,
            persistentStorage);
        stableCache = new StableTemplateCache(this);
        threadPool = new ThreadPoolExecutor(
            config.workers(),
            config.workers(),
            1,
            TimeUnit.MINUTES,
            new LifoWaitBlockingQueue<>(config.connections()),
            new NamedThreadFactory(
                getThreadGroup(),
                getName() + "-W-",
                true));

        register(
            new Pattern<>("/route", false),
            new DelegatedHttpAsyncRequestHandler<>(
                new RouteHandler(this),
                this,
                threadPool),
            false);
        register(
            new Pattern<>("/save_template", false),
            new SaveTemplateHandler(this),
            false);
        closeChain.add(new ExecutorServiceCloser(threadPool));
        registerStater(new ThreadPoolStater(threadPool, "heavy-ops-"));
        register(
            new Pattern<>("/searchLocal", false),
            new LocalMatchHandler(this, threadPool),
            false);
        register(
            new Pattern<>("/index", false),
            new DelegatedHttpAsyncRequestHandler<>(
                new Indexer(this, threadPool),
                this,
                threadPool),
            false);
        SimpleForwardHandler luneneIndexForward = new SimpleForwardHandler(
            luceneIndexClient,
            config.luceneIndexConfig().host(),
            this);
        register(
            new Pattern<>("/getQueueId", false),
            luneneIndexForward,
            false);
        register(new Pattern<>("/lags", false), luneneIndexForward, false);
        register(new Pattern<>("/update", false), luneneIndexForward, false);

        resultFound = new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                resultFound,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "result-template-found_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "result-template-total_ammm",
                        CountAggregatorFactory.INSTANCE))));

        unstableCacheHits = new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                unstableCacheHits,
                new EnumStaterFactory<>(
                    result -> result + "_ammm",
                    AsyncCacheResultType.values(),
                    "unstable-cache-hit-type-",
                    "cache",
                    "unstable cache hit type",
                    null,
                    1)));

        stableCacheHits = new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                stableCacheHits,
                new EnumStaterFactory<>(
                    result -> result + "_ammm",
                    AsyncCacheResultType.values(),
                    "stable-cache-hit-type-",
                    "cache",
                    "stable cache hit type",
                    null,
                    1)));
    }

    public AsyncClient producerAsyncClient() {
        return producerAsyncClient;
    }

    public AsyncClient luceneIndexClient() {
        return luceneIndexClient;
    }

    public AsyncClient luceneSearchClient() {
        return luceneSearchClient;
    }

    public ThreadPoolExecutor threadPool() {
        return threadPool;
    }

    public StableTemplateCache stableCache() {
        return stableCache;
    }

    public UnstableTemplateCache unstableCache() {
        return unstableCache;
    }

    public void resultAchieved(boolean templateFound) {
        resultFound.accept(templateFound ? 1L : 0L);
    }

    public Consumer<AsyncCacheResultType> unstableCacheHits() {
        return unstableCacheHits;
    }

    public Consumer<AsyncCacheResultType> stableCacheHits() {
        return stableCacheHits;
    }
}
