package ru.yandex.mail.so.templatemaster;

import java.util.PriorityQueue;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.collection.LongList;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.mail.so.templatemaster.cache.UnstableTemplateCache;
import ru.yandex.mail.so.templatemaster.config.EvictionConfig;
import ru.yandex.mail.so.templatemaster.templates.UnstableTemplate;

public class UnstableTemplateQueue {
    // Balance quality vs maximum time. Time <
    // < indexing-tags-cutoff-threshold ^ 2 * MAXIMUM_PRECISE_DISTANCE_COUNT =
    // 8192 ^ 2 * 50 = 3.4e9
    private final static int MAXIMUM_PRECISE_DISTANCE_COUNT = 50;

    private final AtomicInteger weight = new AtomicInteger(40);
    private final UnstableTemplateCache cache;
    private final EvictionConfig evictionConfig;
    private final int hitsToStabilize;
    private Node first = null;
    private Node last = null;

    public UnstableTemplateQueue(
        final UnstableTemplateCache cache,
        final EvictionConfig evictionConfig,
        final int hitsToStabilize)
    {
        this.cache = cache;
        this.evictionConfig = evictionConfig;
        this.hitsToStabilize = hitsToStabilize;
    }

    public void add(UnstableTemplate e) {
        weight.addAndGet(40 + e.weight());
        push_front(new Node(e, null, null));
    }

    private void push_front(Node n) {
        if (first == null) {
            n.next = n.prev = null;
            last = first = n;
        } else {
            n.prev = first;
            n.next = null;
            first.next = n;
            first = n;
        }
    }

    public void remove(Node node) {
        if (node == first) {
            first = node.prev;
            if (node == last) {
                last = null;
            } else {
                first.next = null;
            }
        } else {
            node.next.prev = node.prev;
            if (node == last) {
                last = node.next;
            } else {
                node.prev.next = node.next;
            }
        }
    }


    // Eviction policy
    public void prune(
        final ProxySession session,
        final long prefix,
        final long ctime)
    {
        if (last != null) {
            LongList list = new LongList();
            while (last != null
                && last.value.created() + evictionConfig.guaranteedTime() < ctime
                && (last.value.created() + evictionConfig.maximumTime() < ctime
                        || weight.get() > evictionConfig.maximumSize()))
            {
                list.addLong(last.value.url());
                weight.addAndGet(-40 - last.value.weight());
                remove(last);
            }
            if (!list.isEmpty()) {
                cache.persistentStorage().deleteUnstableTemplates(
                    session,
                    "prune",
                    prefix,
                    list);
            }
        }
    }

    public void update(
        UnstableTemplate newTemplate,
        long ctime,
        double digestThreshold,
        ProxySession session,
        FutureCallback<? super Void> callback)
        throws BadRequestException
    {
        long prefix = session.params().getLong("shard");
        prune(session, prefix, ctime);

        Node it = last;

        PriorityQueue<NodeWithDistance> topFastDistance =
            new PriorityQueue<>(MAXIMUM_PRECISE_DISTANCE_COUNT + 1);

        while (it != null) {
            if (it.value.checkMatch(newTemplate.tokens()) != null) {
                it.value.registerHit(newTemplate);
                remove(it);
                if (it.value.hits() >= hitsToStabilize) {
                    session.logger().fine(
                        "Enough hits to stabilize template " + it.value.url());
                    cache.stabilize(it.value, session, callback);
                    return;
                }
                push_front(it);
                session.logger().fine(
                    "Persisting matching template " + it.value.url());
                cache.persist(it.value, session, false, callback);
                return;
            }

            double itsFastDistance = it.value.fastDistance(newTemplate);
            if (itsFastDistance <= digestThreshold
                && (topFastDistance.size() < MAXIMUM_PRECISE_DISTANCE_COUNT
                    || topFastDistance.peek().fastDistance() > itsFastDistance))
            {
                topFastDistance.add(new NodeWithDistance(it, itsFastDistance));
                if (topFastDistance.size() > MAXIMUM_PRECISE_DISTANCE_COUNT) {
                    topFastDistance.poll();
                }
            }

            it = it.next;
        }

        int size = topFastDistance.size();
        NodeWithDistance[] candidates = new NodeWithDistance[size];
        for (int i = 0; i < size; ++i) {
            candidates[i] = topFastDistance.poll();
        }
        Node best = null;
        UnstableTemplate.PreciseDistance bestDistance =
            new UnstableTemplate.PreciseDistance(digestThreshold, null);

        for (int i = candidates.length - 1; i >= 0; --i) {
            // fastDistance is optimistic
            if (candidates[i].fastDistance() >= bestDistance.distance()) {
                continue;
            }
            UnstableTemplate.PreciseDistance distance =
                candidates[i].node().value.preciseDistance(newTemplate);
            if (bestDistance.distance() > distance.distance()) {
                best = candidates[i].node();
                bestDistance = distance;
            }
        }

        if (best != null) {
            remove(best);
            cache.persistentStorage().deleteUnstableTemplates(
                session,
                "replace-best",
                prefix,
                LongList.of(best.value.url()));
            weight.addAndGet(-40 - best.value.weight());
            UnstableTemplate merged = UnstableTemplate.merge(
                best.value,
                newTemplate,
                bestDistance.commonTokens());
            if (merged.hits() >= hitsToStabilize) {
                session.logger().fine(
                    "Enough hits to stabilize merged template "
                    + merged.url());
                cache.stabilize(merged, session, callback);
                return;
            }
            add(merged);
            session.logger().fine("Persisting best template " + merged.url());
            cache.persist(merged, session, true, callback);
        } else {
            add(newTemplate);
            session.logger().fine("Persisting template " + newTemplate.url());
            cache.persist(newTemplate, session, true, callback);
        }
    }

    public int weight() {
        return weight.get();
    }

    // Cause there no LinkedList with public node or other fast pointers
    public static class Node {
        protected final UnstableTemplate value;
        protected Node prev;
        protected Node next;

        public Node(
            UnstableTemplate value,
            Node prev,
            Node next)
        {
            this.value = value;
            this.prev = prev;
            this.next = next;
        }
    }

    private static class NodeWithDistance
        implements Comparable<NodeWithDistance>
    {
        private final Node node;
        private final double fastDistance;

        private NodeWithDistance(Node node, double fastDistance) {
            this.node = node;
            this.fastDistance = fastDistance;
        }

        @Override
        public int compareTo(NodeWithDistance rhs) {
            return Double.compare(rhs.fastDistance, fastDistance);
        }

        public Node node() {
            return node;
        }

        public double fastDistance() {
            return fastDistance;
        }
    }
}
