package ru.yandex.mail.so.templatemaster.cache;

import com.googlecode.concurrentlinkedhashmap.Weigher;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.cache.async.AsyncCache;
import ru.yandex.cache.async.ConcurrentLinkedHashMapAsyncStorage;
import ru.yandex.cache.async.ConcurrentLinkedHashMapAsyncStorageStater;
import ru.yandex.cache.async.ConstAsyncCacheTtlCalculator;
import ru.yandex.cache.async.StringWeigher;
import ru.yandex.cache.async.UnwrappingFutureCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.NByteArrayEntityGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.mail.so.templatemaster.TemplateMaster;
import ru.yandex.mail.so.templatemaster.UnstableTemplateQueue;
import ru.yandex.mail.so.templatemaster.searching.RouteHandler;
import ru.yandex.mail.so.templatemaster.storage.LuceneStableAsyncIndexer;
import ru.yandex.mail.so.templatemaster.storage.LuceneUnstableAsyncIndexer;
import ru.yandex.mail.so.templatemaster.storage.LuceneUnstableAsyncLoader;
import ru.yandex.mail.so.templatemaster.templates.StableTemplate;
import ru.yandex.mail.so.templatemaster.templates.UnstableTemplate;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.util.timesource.TimeSource;

public class UnstableTemplateCache {
    private final static long CACHE_TTL = 3600000L;
    private final LuceneUnstableAsyncIndexer persistentStorage;
    private final TemplateMaster server;
    private final ConcurrentLinkedHashMapAsyncStorage
        <String, UnstableTemplateQueue>
        cacheStorage;
    private final AsyncCache<
        String,
        UnstableTemplateQueue,
        ProxySession,
        RuntimeException> cache;

    public UnstableTemplateCache(
        final TemplateMaster server,
        final LuceneUnstableAsyncIndexer persistentStorage)
    {
        this.server = server;
        this.persistentStorage = persistentStorage;
        cacheStorage =
            new ConcurrentLinkedHashMapAsyncStorage<>(
                server.config().unstableTemplatesCache(),
                server.config().workers(),
                StringWeigher.INSTANCE,
                QueueWeigher.INSTANCE);

        cache =
            new AsyncCache<>(
                cacheStorage,
                new ConstAsyncCacheTtlCalculator(CACHE_TTL),
                new LuceneUnstableAsyncLoader(server, this));
        server.registerStater(
            new ConcurrentLinkedHashMapAsyncStorageStater(
                cacheStorage,
                false,
                "unstable-cache-status-",
                "cache",
                null));
    }

    public void reInsert(String key, UnstableTemplateQueue value) {
        cacheStorage.put(
            key,
            value,
            CACHE_TTL,
            EmptyFutureCallback.instance());
    }

    public void persist(
        UnstableTemplate value,
        ProxySession session,
        boolean saveTokens,
        FutureCallback<? super Void> callback)
    {
        ///TODO : true -> saveTokens
        persistentStorage.putUnstableTemplate(value, session, true, callback);
    }

    public LuceneUnstableAsyncIndexer persistentStorage() {
        return persistentStorage;
    }

    public void get(
        String domain,
        ProxySession session,
        FutureCallback<UnstableTemplateQueue> callback)
    {
        cache.get(
            domain,
            session,
            new UnwrappingFutureCallback<>(
                callback,
                server.unstableCacheHits(),
                session.logger()));
    }

    public void stabilize(
        UnstableTemplate template,
        ProxySession session,
        FutureCallback<? super Void> callback)
    {
        QueryConstructor qc = new QueryConstructor("/save_template?", false);
        try {
            String domain = session.params().getString("domain");
            int shard = RouteHandler.calculateShardByDomain(
                domain,
                server.config().numShards());
            qc.append("service", RouteHandler.SERVICE);
            qc.append("shard", shard);
            qc.append("drop_domain", domain);

            NByteArrayEntityGenerator request =
                LuceneStableAsyncIndexer.prepareIndexRequest(
                    new StableTemplate(
                        template.tokens(),
                        template.url(),
                        template.attributes(),
                        TimeSource.INSTANCE.currentTimeMillis()),
                    domain,
                    shard);

            AsyncClient client =
                server.producerAsyncClient().adjust(session.context());
            client.execute(
                server.config().producerAsyncClientConfig().host(),
                new BasicAsyncRequestProducerGenerator(
                    qc.toString(),
                    request),
                EmptyAsyncConsumerFactory.ANY_GOOD,
                session.listener().createContextGeneratorFor(client),
                callback);
        } catch (BadRequestException e) {
            callback.failed(new RuntimeException("stabilizing failed", e));
        }
    }

    private enum QueueWeigher implements Weigher<UnstableTemplateQueue> {
        INSTANCE;

        @Override
        public int weightOf(final UnstableTemplateQueue queue) {
            return 210 + queue.weight();
        }
    }
}

