package ru.yandex.mail.so.templatemaster.config;

import ru.yandex.http.config.HttpHostConfig;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.LongMemorySizeParser;
import ru.yandex.search.proxy.universal.AbstractUniversalSearchProxyConfigBuilder;

public abstract class AbstractTemplateMasterConfigBuilder
    <T extends AbstractTemplateMasterConfigBuilder<T>>
    extends AbstractUniversalSearchProxyConfigBuilder<T>
    implements TemplateMasterConfig
{

    private HttpHostConfigBuilder luceneIndexConfig;
    private HttpHostConfigBuilder luceneSearchConfig;
    private HttpHostConfigBuilder producerAsyncClientConfig;
    private int numShards;
    private long searchFailoverDelay;
    private long stableTemplatesCache;
    private long unstableTemplatesCache;
    private EvictionConfig eviction;
    private double matchingThreshold;
    private int hitsToStabilize;
    private int indexingTagsCutoffThreshold;

    protected AbstractTemplateMasterConfigBuilder(TemplateMasterConfig config)
    {
        super(config);
        luceneIndexConfig(config.luceneIndexConfig());
        luceneSearchConfig(config.luceneSearchConfig());
        producerAsyncClientConfig(config.producerAsyncClientConfig());
        numShards(config.numShards());
        searchFailoverDelay(config.searchFailoverDelay());
        stableTemplatesCache(config.stableTemplatesCache());
        unstableTemplatesCache(config.stableTemplatesCache());
        eviction(config.eviction());
        matchingThreshold(config.matchingThreshold());
        hitsToStabilize(config.hitsToStabilize());
        indexingTagsCutoffThreshold(config.indexingTagsCutoffThreshold());
    }

    protected AbstractTemplateMasterConfigBuilder(
        final IniConfig config,
        final TemplateMasterConfig defaults)
        throws ConfigException
    {
        super(config, defaults);
        producerAsyncClientConfig = new HttpHostConfigBuilder(
            config.section("producer-async-client"),
            defaults.producerAsyncClientConfig());
        luceneIndexConfig = new HttpHostConfigBuilder(
            config.section("lucene-index"),
            defaults.luceneIndexConfig());
        luceneSearchConfig = new HttpHostConfigBuilder(
            config.section("lucene-search"),
            defaults.luceneSearchConfig());
        numShards =
            config.getInt("searchmap.num-shards", defaults.numShards());
        searchFailoverDelay =
            config.getLong(
                "search.failover-delay",
                defaults.searchFailoverDelay());
        stableTemplatesCache =
            config.get(
                "stable-template-cache",
                defaults.stableTemplatesCache(),
                LongMemorySizeParser.INSTANCE);
        unstableTemplatesCache =
            config.get(
                "unstable-template-cache",
                defaults.stableTemplatesCache(),
                LongMemorySizeParser.INSTANCE);
        eviction = new EvictionConfig(
            config.sectionOrDefault("unstable-eviction", IniConfig.empty()),
            defaults.eviction());
        matchingThreshold = config.getDouble(
            "matching-threshold",
            defaults.matchingThreshold());
        hitsToStabilize =
            config.getInt("hits-to-stabilize", defaults.hitsToStabilize());
        indexingTagsCutoffThreshold =
            config.getInt("indexing-tags-cutoff-threshold");
    }

    public T producerAsyncClientConfig(
        HttpHostConfig producerAsyncClientConfig)
    {
        this.producerAsyncClientConfig =
            new HttpHostConfigBuilder(producerAsyncClientConfig);
        return self();
    }

    public T luceneIndexConfig(HttpHostConfig luceneIndexConfig) {
        this.luceneIndexConfig =
            new HttpHostConfigBuilder(luceneIndexConfig);
        return self();
    }

    public T luceneSearchConfig(HttpHostConfig luceneSearchConfig) {
        this.luceneSearchConfig =
            new HttpHostConfigBuilder(luceneSearchConfig);
        return self();
    }

    public T numShards(int numShards) {
        this.numShards = numShards;
        return self();
    }

    public T searchFailoverDelay(long searchFailoverDelay) {
        this.searchFailoverDelay = searchFailoverDelay;
        return self();
    }

    public T stableTemplatesCache(long stableTemplatesCache) {
        this.stableTemplatesCache = stableTemplatesCache;
        return self();
    }

    public T unstableTemplatesCache(long unstableTemplatesCache) {
        this.unstableTemplatesCache = unstableTemplatesCache;
        return self();
    }

    public T eviction(EvictionConfig eviction) {
        this.eviction = eviction;
        return self();
    }

    public T matchingThreshold(double matchingThreshold) {
        this.matchingThreshold = matchingThreshold;
        return self();
    }

    public T hitsToStabilize(int hitsToStabilize) {
        this.hitsToStabilize = hitsToStabilize;
        return self();
    }

    public T indexingTagsCutoffThreshold(int indexingTagsCutoffThreshold) {
        this.indexingTagsCutoffThreshold = indexingTagsCutoffThreshold;
        return self();
    }

    @Override
    public HttpHostConfig producerAsyncClientConfig() {
        return producerAsyncClientConfig;
    }

    @Override
    public HttpHostConfigBuilder luceneIndexConfig() {
        return luceneIndexConfig;
    }

    @Override
    public HttpHostConfigBuilder luceneSearchConfig() {
        return luceneSearchConfig;
    }

    @Override
    public int numShards() {
        return numShards;
    }

    @Override
    public long searchFailoverDelay() {
        return searchFailoverDelay;
    }

    @Override
    public long stableTemplatesCache() {
        return stableTemplatesCache;
    }

    @Override
    public EvictionConfig eviction() {
        return this.eviction;
    }

    @Override
    public long unstableTemplatesCache() {
        return unstableTemplatesCache;
    }

    @Override
    public double matchingThreshold() {
        return matchingThreshold;
    }

    @Override
    public int hitsToStabilize() {
        return hitsToStabilize;
    }

    @Override
    public int indexingTagsCutoffThreshold() {
        return indexingTagsCutoffThreshold;
    }
}
