package ru.yandex.mail.so.templatemaster.indexing;

import java.util.Arrays;
import java.util.concurrent.Executor;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.function.ByteArrayProcessable;
import ru.yandex.function.ByteBufferFactory;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.BasicProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ExecutorFutureCallback;
import ru.yandex.http.util.nio.ByteArrayProcessableAsyncConsumer;
import ru.yandex.mail.so.templatemaster.TemplateMaster;
import ru.yandex.mail.so.templatemaster.UnstableTemplateQueue;
import ru.yandex.mail.so.templatemaster.templates.BaseTemplate;
import ru.yandex.mail.so.templatemaster.templates.UnstableTemplate;

public class Indexer implements HttpAsyncRequestHandler<ByteArrayProcessable> {
    private final TemplateMaster server;
    private final int tokensCutoff;
    private final Executor threadPool;


    public Indexer(
        TemplateMaster server,
        Executor threadPool)
    {
        this.server = server;
        tokensCutoff = server.config().indexingTagsCutoffThreshold();
        this.threadPool = threadPool;
    }

    @Override
    public HttpAsyncRequestConsumer<ByteArrayProcessable> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }
        return new ByteArrayProcessableAsyncConsumer();
    }

    @Override
    public void handle(
        final ByteArrayProcessable body,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session =
            new BasicProxySession(server, exchange, context);
        String domain = session.params().getString("domain");

        long[] requestTokens =
            BaseTemplate.hashesFromBytes(
                body.processWith(ByteBufferFactory.INSTANCE));

        if (requestTokens.length > tokensCutoff) {
            requestTokens = Arrays.copyOf(requestTokens, tokensCutoff + 1);
            requestTokens[tokensCutoff] = BaseTemplate.SEP_TOKEN;
        }

        server.unstableCache().get(
            domain,
            session,
            new ExecutorFutureCallback<>(
                new IndexingCallback(
                    session,
                    UnstableTemplate.fromSingle(
                        requestTokens,
                        session.params().getLong("ctime"),
                        '[' + session.params().getString("attributes") + ']'),
                    new BasicProxySessionCallback(session)),
                session,
                threadPool));
    }

    private class IndexingCallback
        extends AbstractFilterFutureCallback<UnstableTemplateQueue, Void>
    {
        private final UnstableTemplate current;
        private final ProxySession session;

        protected IndexingCallback(
            ProxySession session,
            UnstableTemplate current,
            BasicProxySessionCallback callback)
        {
            super(callback);
            this.current = current;
            this.session = session;
        }

        @Override
        public void completed(UnstableTemplateQueue queue) {
            session.logger().fine("Got unstable queue, updating its data");
            String domain;
            try {
                domain = session.params().getString("domain");
                queue.update(
                    current,
                    session.params().getLong("ctime"),
                    server.config().matchingThreshold(),
                    session,
                    callback);
            } catch (BadRequestException e) {
                failed(e);
                return;
            }
            // cause weight changed
            server.unstableCache().reInsert(domain, queue);
        }
    }

}
