package ru.yandex.mail.so.templatemaster.searching;

import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.charset.Decoder;
import ru.yandex.compress.GzipInputStream;
import ru.yandex.digest.Fnv;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ByteArrayProcessableWithContentType;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.ByteArrayProcessableWithContentTypeAsyncConsumer;
import ru.yandex.http.util.nio.NByteArrayEntityAsyncConsumerFactory;
import ru.yandex.http.util.nio.NByteArrayEntityGenerator;
import ru.yandex.io.ByteArrayInputStreamFactory;
import ru.yandex.io.IOStreamUtils;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.mail.so.templatemaster.TemplateMaster;
import ru.yandex.mail.so.templatemaster.templates.BaseTemplate;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.ShardPrefix;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;

public class RouteHandler
    implements HttpAsyncRequestHandler<ByteArrayProcessableWithContentType>
{
    public static final String SERVICE = "template_master";

    protected final TemplateMaster server;

    public RouteHandler(TemplateMaster server) {
        this.server = server;
    }

    public static int calculateShardByDomain(String domain, int numShards) {
        return Math.floorMod(domain.hashCode(), numShards);
    }

    @Override
    public HttpAsyncRequestConsumer<ByteArrayProcessableWithContentType>
        processRequest(final HttpRequest request, final HttpContext context)
        throws HttpException
    {
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }
        return new ByteArrayProcessableWithContentTypeAsyncConsumer();
    }

    @Override
    public void handle(
        final ByteArrayProcessableWithContentType payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        String html = unwrapMaybeGzipped(payload);
        ProxySession session =
            new BasicProxySession(server, exchange, context);

        String[] tokens = tokenize(html);
        Map<Long, String> hashToToken = new HashMap<>();
        long[] hashes = new long[tokens.length];
        for (int i = 0; i < tokens.length; ++i) {
            hashes[i] = Fnv.fnv64(tokens[i]);
            hashToToken.put(hashes[i], tokens[i]);
        }
        routeRequest(
            BaseTemplate.hashesToBytes(hashes),
            session,
            new HashesDecodingCallback(session, hashToToken));
    }

    public static String[] tokenize(final String html) {
        List<String> result = new ArrayList<>(1024);
        int prev = 0;
        while (true) {
            int start = html.indexOf('<', prev);
            if (start == -1) {
                break;
            }
            if (start > prev) {
                result.add(html.substring(prev, start));
            }
            prev = html.indexOf('>', start + 1) + 1;
            result.add(html.substring(start, prev));
        }
        if (prev < html.length()) {
            result.add(html.substring(prev));
        }
        return result.toArray(new String[result.size()]);
    }

    public void routeRequest(
        final byte[] payload,
        final ProxySession session,
        final FutureCallback<HttpEntity> callback)
        throws HttpException
    {
        String domain = session.params().getString("domain");
        int shard =
            calculateShardByDomain(domain, server.config().numShards());

        UniversalSearchProxyRequestContext searchProxyContext =
            new PlainUniversalSearchProxyRequestContext(
                new User(SERVICE, new ShardPrefix(shard)),
                null,
                true,
                server.searchClient().adjust(session.context()),
                session.logger());
        QueryConstructor query = new QueryConstructor("/searchLocal?", false);
        query.append("shard", shard);
        query.append("domain", domain);
        query.copyIfPresent(session.params(), "attributes");
        query.copyIfPresent(session.params(), "dry-run");
        server.sequentialRequest(
            session,
            searchProxyContext,
            new BasicAsyncRequestProducerGenerator(
                query.toString(),
                new NByteArrayEntityGenerator(
                    payload,
                    ContentType.APPLICATION_OCTET_STREAM,
                    null)),
            server.config().searchFailoverDelay(),
            true,
            NByteArrayEntityAsyncConsumerFactory.OK,
            session.listener()
                .createContextGeneratorFor(searchProxyContext.client()),
            callback);
    }

    public String unwrapMaybeGzipped(
        final ByteArrayProcessableWithContentType payload)
        throws IOException
    {
        Decoder decoder = new Decoder(payload.contentType().getCharset());
        Header contentEncodingHeader = payload.contentEncodingHeader();
        if (contentEncodingHeader != null
            && "gzip".equals(contentEncodingHeader.getValue()))
        {
            IOStreamUtils.consume(
                new GzipInputStream(
                    payload.data().processWith(
                        ByteArrayInputStreamFactory.INSTANCE)))
                .processWith(decoder);
        } else {
            payload.data().processWith(decoder);
        }
        return decoder.toString();
    }

    /// Decodes delta (turns hashes back to tokens)
    private static class HashesDecodingCallback extends
        AbstractProxySessionCallback<HttpEntity>
    {
        private final Map<Long, String> hashToToken;


        protected HashesDecodingCallback(
            ProxySession session,
            Map<Long, String> hashToToken)
        {
            super(session);
            this.hashToToken = hashToToken;
        }

        @Override
        public void completed(HttpEntity entity) {
            try {
                JsonMap response =
                    TypesafeValueContentHandler.parse(
                        new InputStreamReader(
                            entity.getContent(),
                            StandardCharsets.UTF_8))
                        .asMap();
                JsonList hashDelta = response.getListOrNull("delta");
                if (hashDelta == null) {
                    session.response(
                        HttpStatus.SC_OK,
                        TemplatesMatchingCallback.NOT_FOUND);
                    return;
                }
                JsonList stringDelta =
                    new JsonList(hashDelta.containerFactory());
                for (JsonObject subsequence : hashDelta) {
                    JsonList stringSubsequence =
                        new JsonList(stringDelta.containerFactory());
                    for (JsonObject hash : subsequence.asList()) {
                        stringSubsequence.add(
                            new JsonString(hashToToken.get(hash.asLong())));
                    }
                    stringDelta.add(stringSubsequence);
                }
                response.put("delta", stringDelta);
                session.response(
                    HttpStatus.SC_OK,
                    new NStringEntity(
                        JsonTypeExtractor.DOLLAR.extract(session.params())
                            .toString(response),
                        ContentType.APPLICATION_JSON));
            } catch (JsonException | IOException | BadRequestException e) {
                failed(e);
            }
        }
    }
}
