package ru.yandex.mail.so.templatemaster.searching;

import java.io.IOException;
import java.nio.charset.StandardCharsets;

import org.apache.http.HttpStatus;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.collection.LongList;
import ru.yandex.function.ByteArrayProcessable;
import ru.yandex.function.ByteBufferFactory;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.mail.so.templatemaster.TemplateMaster;
import ru.yandex.mail.so.templatemaster.storage.LuceneStableAsyncIndexer;
import ru.yandex.mail.so.templatemaster.templates.BaseTemplate;
import ru.yandex.mail.so.templatemaster.templates.StableTemplate;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.util.timesource.TimeSource;

class TemplatesMatchingCallback
    extends AbstractProxySessionCallback<StableTemplate[]>
{
    public static final ByteArrayEntity NOT_FOUND =
        new ByteArrayEntity(
            "{\"status\": \"NotFound\"}".getBytes(StandardCharsets.US_ASCII),
            ContentType.APPLICATION_JSON);
    private static final long MIN_UPDATE_INTERVAL = 300000L;

    private final TemplateMaster server;
    private final ByteArrayProcessable body;
    private final String domain;

    protected TemplatesMatchingCallback(
        final ProxySession session,
        final TemplateMaster server,
        final ByteArrayProcessable body,
        final String domain)
    {
        super(session);
        this.server = server;
        this.body = body;
        this.domain = domain;
    }

    @Override
    public void completed(StableTemplate[] stableTemplates) {
        long totalScannedSize = 0;
        long startingTime = TimeSource.INSTANCE.currentTimeMillis();
        String attributes = session.params().getString("attributes", null);
        long[] requestTokens =
            BaseTemplate.hashesFromBytes(
                body.processWith(ByteBufferFactory.INSTANCE));

        if (stableTemplates.length == 0) {
            session.response(HttpStatus.SC_OK, NOT_FOUND);
            scheduleIndexing(attributes);
            return;
        }

        for (StableTemplate stable : stableTemplates) {
            LongList rawDelta = stable.checkMatch(requestTokens);
            totalScannedSize += stable.tokens().length;
            if (rawDelta == null) {
                continue;
            }
            JsonList delta = new JsonList(BasicContainerFactory.INSTANCE);
            JsonList entry = new JsonList(BasicContainerFactory.INSTANCE);
            for (long token : rawDelta) {
                if (token == BaseTemplate.SEP_TOKEN) {
                    delta.add(entry);
                    entry = new JsonList(BasicContainerFactory.INSTANCE);
                } else {
                    entry.add(new JsonLong(token));
                }
            }
            delta.add(entry);

            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
                writer.startObject();
                writer.key("status");
                writer.value("FoundInDb");
                writer.key("stable_sign");
                writer.value(stable.stableSign());
                writer.key("attributes");
                writer.jsonValue(stable.attributes());
                writer.key("delta");
                delta.writeValue(writer);
                writer.endObject();
            } catch (IOException e) {
                failed(e);
                return;
            }

            session.logger().fine(
                "Template found for domain \"" + domain
                + "\", scanned templates with total size of "
                + totalScannedSize + " in "
                + (TimeSource.INSTANCE.currentTimeMillis() - startingTime)
                + " ms");

            session.response(
                HttpStatus.SC_OK,
                new NStringEntity(
                    sbw.toString(),
                    ContentType.APPLICATION_JSON));
            registerMatch(stable);
            return;
        }

        session.logger().fine(
            "Not found template for domain \"" + domain
            + "\", scanned " + stableTemplates.length
            + " templates with total size of " + totalScannedSize + " in "
            + (TimeSource.INSTANCE.currentTimeMillis() - startingTime)
            + " ms");

        session.response(HttpStatus.SC_OK, NOT_FOUND);
        scheduleIndexing(attributes);
    }

    private void scheduleIndexing(String attributes) {
        server.resultAchieved(false);

        QueryConstructor qc;
        try {
            if (session.params().getBoolean("dry-run", false)) {
                session.response(HttpStatus.SC_OK, NOT_FOUND);
                return;
            }

            long receivedTime = TimeSource.INSTANCE.currentTimeMillis();
            qc = new QueryConstructor("/index?", false);
            qc.append("service", RouteHandler.SERVICE);
            qc.copy(session.params(), "shard");
            qc.append("domain", domain);
            qc.append("ctime", receivedTime);
            qc.append("attributes", attributes);
        } catch (BadRequestException e) {
            failed(e);
            return;
        }

        fireAndForget(
            new BasicAsyncRequestProducerGenerator(
                qc.toString(),
                body,
                ContentType.APPLICATION_OCTET_STREAM));
    }

    private void registerMatch(final StableTemplate template) {
        server.resultAchieved(true);
        long now = TimeSource.INSTANCE.currentTimeMillis();
        if (now - template.lastAccess() > MIN_UPDATE_INTERVAL) {
            template.lastAccess(now);
            String shard = session.params().getString("shard", "");
            fireAndForget(
                new BasicAsyncRequestProducerGenerator(
                    "/update?service=" + RouteHandler.SERVICE
                    + "&shard=" + shard,
                LuceneStableAsyncIndexer.prepareUpdateLastAccessRequest(
                    template,
                    shard)));
        }
    }

    private void fireAndForget(BasicAsyncRequestProducerGenerator request) {
        AsyncClient client =
            server.producerAsyncClient().adjust(session.context());
        client.execute(
            server.config().producerAsyncClientConfig().host(),
            request,
            EmptyAsyncConsumerFactory.ANY_GOOD,
            session.listener().createContextGeneratorFor(client),
            EmptyFutureCallback.INSTANCE);
    }
}
