package ru.yandex.mail.so.templatemaster.storage;

import java.util.Arrays;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.cache.async.AsyncLoader;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ExecutorFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.ByteArrayProcessableWithContentTypeAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.parser.JsonException;
import ru.yandex.mail.so.templatemaster.TemplateMaster;
import ru.yandex.mail.so.templatemaster.templates.StableTemplate;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.request.util.SearchRequestText;

/**
 * Fetches stable templates by domain from Lucene
 * Requires "shard" query param
 */
public class LuceneStableAsyncLoader
    implements AsyncLoader<String, StableTemplate[], ProxySession>
{
    private final TemplateMaster server;

    public LuceneStableAsyncLoader(final TemplateMaster server) {
        this.server = server;
    }

    @Override
    public void load(
        final String domain,
        final ProxySession session,
        final FutureCallback<? super StableTemplate[]> callback)
    {
        AsyncClient client =
            server.luceneSearchClient().adjustZooHeaders(session.context());
        QueryConstructor qc = new QueryConstructor(
            "/search?json-type=dollar&get=tokens,url,attributes,last_access"
            + "&memory-limit=100m&sort=last_access");
        try {
            qc.append("prefix", session.params().getString("shard"));
            qc.append(
                "text",
                "group:stable_" + SearchRequestText.fullEscape(domain, false));
        } catch (BadRequestException e) {
            callback.failed(e);
            return;
        }
        client.execute(
            server.config().luceneSearchConfig().host(),
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            ByteArrayProcessableWithContentTypeAsyncConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            new ExecutorFutureCallback<>(
                new StableTemplatesParser(callback),
                session,
                server.threadPool()));
    }

    private static class StableTemplatesParser
        extends TemplatesParserBase<StableTemplate, StableTemplate[]>
    {
        private static final char[] LAST_ACCESS = "last_access".toCharArray();

        private long lastAccess = 0L;

        StableTemplatesParser(
            final FutureCallback<? super StableTemplate[]> callback)
        {
            super(callback);
        }

        @Override
        protected void commitTemplate() {
            templates.add(
                new StableTemplate(
                    tokens,
                    url,
                    attributes,
                    lastAccess));
            lastAccess = 0L;
        }

        @Override
        protected void done() {
            callback.completed(
                templates.toArray(new StableTemplate[templates.size()]));
        }

        @Override
        public void key(
            final char[] buf,
            final int off,
            final int len,
            final boolean eol)
            throws JsonException
        {
            if (eol
                && len == 11
                && state == State.DOC
                && Arrays.equals(
                    buf, off, off + len,
                    LAST_ACCESS, 0, LAST_ACCESS.length))
            {
                state = State.LAST_ACCESS;
            } else {
                super.key(buf, off, len, eol);
            }
        }

        @Override
        public void value(
            final char[] buf,
            final int off,
            final int len,
            final boolean eol)
            throws JsonException
        {
            if (eol && state == State.LAST_ACCESS) {
                state = State.DOC;
                lastAccess = Long.parseLong(new String(buf, off, len));
            } else {
                super.value(buf, off, len, eol);
            }
        }

        @Override
        public void value(final long value) throws JsonException {
            if (state == State.LAST_ACCESS) {
                state = State.DOC;
                lastAccess = value;
            } else {
                super.value(value);
            }
        }
    }
}
