package ru.yandex.mail.so.templatemaster.storage;

import java.util.Arrays;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.cache.async.AsyncLoader;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ExecutorFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.ByteArrayProcessableWithContentTypeAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.parser.JsonException;
import ru.yandex.mail.so.templatemaster.TemplateMaster;
import ru.yandex.mail.so.templatemaster.UnstableTemplateQueue;
import ru.yandex.mail.so.templatemaster.cache.UnstableTemplateCache;
import ru.yandex.mail.so.templatemaster.templates.UnstableTemplate;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.request.util.SearchRequestText;

/**
 * Fetches unstable templates by domain from Lucene
 * Requires "shard" query param
 */
public class LuceneUnstableAsyncLoader
    implements AsyncLoader<String, UnstableTemplateQueue, ProxySession>
{
    private static final char[] CREATED = "created".toCharArray();
    private static final char[] HITS = "hits".toCharArray();
    private static final char[] LEN_SUM = "len_sum".toCharArray();

    private final TemplateMaster server;
    private final UnstableTemplateCache cache;

    public LuceneUnstableAsyncLoader(
        final TemplateMaster server,
        final UnstableTemplateCache cache)
    {
        this.server = server;
        this.cache = cache;
    }

    @Override
    public void load(
        String domain,
        ProxySession session,
        FutureCallback<? super UnstableTemplateQueue> callback)
    {
        AsyncClient client =
            server.luceneSearchClient().adjustZooHeaders(session.context());
        QueryConstructor qc = new QueryConstructor(
            "/search?json-type=dollar"
            + "&get=tokens,url,attributes,created,hits,len_sum"
            + "&memory-limit=100m&sort=created&asc");
        try {
            qc.append("prefix", session.params().getString("shard"));
            qc.append(
                "text",
                "group:unstable_"
                    + SearchRequestText.fullEscape(domain, false));
        } catch (BadRequestException e) {
            callback.failed(e);
            return;
        }
        client.execute(
            server.config().luceneSearchConfig().host(),
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            ByteArrayProcessableWithContentTypeAsyncConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            new ExecutorFutureCallback<>(
                new UnstableTemplatesParser(callback),
                session,
                server.threadPool()));
    }

    private class UnstableTemplatesParser
        extends TemplatesParserBase<UnstableTemplate, UnstableTemplateQueue>
    {
        private long created = 0L;
        private int hits = 0;
        private int lenSum = 0;

        UnstableTemplatesParser(
            final FutureCallback<? super UnstableTemplateQueue> callback)
        {
            super(callback);
        }

        @Override
        protected void commitTemplate() {
            templates.add(
                new UnstableTemplate(
                    tokens,
                    url,
                    attributes,
                    created,
                    hits,
                    lenSum));
            created = 0L;
            hits = 0;
            lenSum = 0;
        }

        @Override
        protected void done() {
            int size = templates.size();
            UnstableTemplateQueue res = new UnstableTemplateQueue(
                cache,
                server.config().eviction(),
                server.config().hitsToStabilize());
            for (int i = 0; i < size; ++i) {
                res.add(templates.get(i));
            }

            callback.completed(res);
        }

        @Override
        public void key(
            final char[] buf,
            final int off,
            final int len,
            final boolean eol)
            throws JsonException
        {
            if (eol && state == State.DOC) {
                if (len == 4
                    && Arrays.equals(
                        buf, off, off + len,
                        HITS, 0, HITS.length))
                {
                    state = State.HITS;
                    return;
                } else if (len == 7) {
                    if (Arrays.equals(
                        buf, off, off + len,
                        CREATED, 0, CREATED.length))
                    {
                        state = State.CREATED;
                        return;
                    } else if (Arrays.equals(
                        buf, off, off + len,
                        LEN_SUM, 0, LEN_SUM.length))
                    {
                        state = State.LEN_SUM;
                        return;
                    }
                }
            }
            super.key(buf, off, len, eol);
        }

        @Override
        public void value(
            final char[] buf,
            final int off,
            final int len,
            final boolean eol)
            throws JsonException
        {
            if (eol) {
                switch (state) {
                    case CREATED:
                        state = State.DOC;
                        created = Long.parseLong(new String(buf, off, len));
                        return;
                    case HITS:
                        state = State.DOC;
                        hits = Integer.parseInt(new String(buf, off, len));
                        return;
                    case LEN_SUM:
                        state = State.DOC;
                        lenSum = Integer.parseInt(new String(buf, off, len));
                        return;
                    default:
                        break;
                }
            }
            super.value(buf, off, len, eol);
        }

        @Override
        public void value(final long value) throws JsonException {
            switch (state) {
                case CREATED:
                    state = State.DOC;
                    created = value;
                    return;
                case HITS:
                    state = State.DOC;
                    hits = (int) value;
                    return;
                case LEN_SUM:
                    state = State.DOC;
                    lenSum = (int) value;
                    return;
                default:
                    break;
            }
            super.value(value);
        }
    }
}
