package ru.yandex.mail.so.templatemaster.storage;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.base64.Base64;
import ru.yandex.base64.Base64Decoder;
import ru.yandex.charset.Decoder;
import ru.yandex.function.ByteBufferFactory;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.ByteArrayProcessableWithContentType;
import ru.yandex.json.parser.ContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.mail.so.templatemaster.templates.BaseTemplate;

public abstract class TemplatesParserBase<T, Q>
    extends AbstractFilterFutureCallback
        <ByteArrayProcessableWithContentType, Q>
    implements ContentHandler
{
    private static final ThreadLocal<Base64Decoder> BASE64_DECODER =
        ThreadLocal.withInitial(() -> new Base64Decoder(Base64.URL));

    private static final char[] HITS_COUNT = "hitsCount".toCharArray();
    private static final char[] HITS_ARRAY = "hitsArray".toCharArray();
    private static final char[] TOKENS = "tokens".toCharArray();
    private static final char[] URL = "url".toCharArray();
    private static final char[] ATTRIBUTES = "attributes".toCharArray();

    protected enum State {
        INITIAL,
        ROOT,
        HITS_COUNT,
        HITS_ARRAY,
        INSIDE_HITS_ARRAY,
        DOC,
        TOKENS,
        URL,
        ATTRIBUTES,
        LAST_ACCESS,
        CREATED,
        HITS,
        LEN_SUM,
        END
    }

    private final Base64Decoder decoder = BASE64_DECODER.get();
    protected final List<T> templates = new ArrayList<>(16);
    protected State state = State.INITIAL;
    protected long[] tokens = null;
    protected long url = 0L;
    protected String attributes = null;

    protected TemplatesParserBase(
        final FutureCallback<? super Q> callback)
    {
        super(callback);
    }

    protected abstract void commitTemplate();

    protected abstract void done();

    @Override
    public void completed(final ByteArrayProcessableWithContentType result) {
        try {
            Decoder decoder =
                new Decoder(result.contentType().getCharset());
            result.data().processWith(decoder);
            JsonParser parser = new JsonParser(this);
            decoder.processWith(parser);
            parser.eof();
            done();
        } catch (IOException | JsonException e) {
            failed(e);
        }
    }

    @Override
    public void startObject() throws JsonException {
        switch (state) {
            case INITIAL:
                state = State.ROOT;
                break;
            case INSIDE_HITS_ARRAY:
                state = State.DOC;
                break;
            default:
                throw new JsonException(
                    "Unexpected '{' while state is " +  state);
        }
    }

    @Override
    public void endObject() throws JsonException {
        switch (state) {
            case ROOT:
                state = State.END;
                break;
            case DOC:
                commitTemplate();
                tokens = null;
                url = 0L;
                attributes = null;
                state = State.INSIDE_HITS_ARRAY;
                break;
            default:
                throw new JsonException(
                    "Unexpected '}' while state is " +  state);
        }
    }

    @Override
    public void startArray() throws JsonException {
        if (state == State.HITS_ARRAY) {
            state = State.INSIDE_HITS_ARRAY;
        } else {
            throw new JsonException(
                "Unexpected '[' while state is " + state);
        }
    }

    @Override
    public void endArray() throws JsonException {
        if (state == State.INSIDE_HITS_ARRAY) {
            state = State.ROOT;
        } else {
            throw new JsonException(
                "Unexpected ']' while state is " + state);
        }
    }

    @Override
    public void key(
        final char[] buf,
        final int off,
        final int len,
        final boolean eol)
        throws JsonException
    {
        if (eol) {
            switch (len) {
                // hitsCount
                // hitsArray
                case 9:
                    if (state == State.ROOT) {
                        if (Arrays.equals(
                            buf, off, off + len,
                            HITS_COUNT, 0, HITS_COUNT.length))
                        {
                            state = State.HITS_COUNT;
                            return;
                        } else if (Arrays.equals(
                            buf, off, off + len,
                            HITS_ARRAY, 0, HITS_ARRAY.length))
                        {
                            state = State.HITS_ARRAY;
                            return;
                        }
                    }
                    break;
                // tokens
                case 6:
                    if (state == State.DOC
                        && Arrays.equals(
                            buf, off, off + len,
                            TOKENS, 0, TOKENS.length))
                    {
                        state = State.TOKENS;
                        return;
                    }
                    break;
                // url
                case 3:
                    if (state == State.DOC
                        && Arrays.equals(
                            buf, off, off + len,
                            URL, 0, URL.length))
                    {
                        state = State.URL;
                        return;
                    }
                    break;
                // attributes
                case 10:
                    if (state == State.DOC
                        && Arrays.equals(
                            buf, off, off + len,
                            ATTRIBUTES, 0, ATTRIBUTES.length))
                    {
                        state = State.ATTRIBUTES;
                        return;
                    }
                    break;
                default:
                    break;
            }
        }
        StringBuilder sb = new StringBuilder("Unexpected key '");
        sb.append(buf, off, len);
        if (!eol) {
            sb.append('…');
        }
        sb.append("' while state is ");
        sb.append(state);
        throw new JsonException(new String(sb));
    }

    @Override
    public void value(
        final char[] buf,
        final int off,
        final int len,
        final boolean eol)
        throws JsonException
    {
        if (eol) {
            switch (state) {
                case HITS_COUNT:
                    state = State.ROOT;
                    return;
                case TOKENS:
                    state = State.DOC;
                    try {
                        decoder.process(buf, off, len);
                        tokens =
                            BaseTemplate.hashesFromBytes(
                                decoder.processWith(
                                    ByteBufferFactory.INSTANCE));
                        return;
                    } catch (IOException e) {
                        // Ignore, will report at the end of function
                    }
                    break;
                case URL:
                    state = State.DOC;
                    // TODO: optimize
                    url = Long.parseLong(new String(buf, off, len));
                    return;
                case ATTRIBUTES:
                    state = State.DOC;
                    attributes = new String(buf, off, len);
                    return;
                default:
                    break;
            }
        }
        StringBuilder sb = new StringBuilder("Unexpected string value '");
        sb.append(buf, off, len);
        if (!eol) {
            sb.append('…');
        }
        sb.append("' while state is ");
        sb.append(state);
        throw new JsonException(new String(sb));
    }

    @Override
    public void value(final long value) throws JsonException {
        switch (state) {
            case HITS_COUNT:
                state = State.ROOT;
                return;
            case URL:
                state = State.DOC;
                url = value;
                return;
            default:
                break;
        }
        throw new JsonException(
            "Unexpected long value " + value + " while state is " + state);
    }

    @Override
    public void value(final double value) throws JsonException {
        throw new JsonException(
            "Unexpected double value " + value + " while state is "
            + state);
    }

    @Override
    public void value(final boolean value) throws JsonException {
        throw new JsonException(
            "Unexpected boolean value " + value + " while state is "
            + state);
    }

    @Override
    public void nullValue() throws JsonException {
        throw new JsonException(
            "Unexpected null value while state is " + state);
    }

    @Override
    public void endObjectEntry() {
    }
}
