package ru.yandex.mail.so.templatemaster.templates;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;

import javax.annotation.Nullable;

import ru.yandex.base64.Base64;
import ru.yandex.base64.Base64Decoder;
import ru.yandex.base64.Base64Encoder;
import ru.yandex.collection.LongList;
import ru.yandex.concurrent.Weighable;
import ru.yandex.function.ByteBufferFactory;

public class BaseTemplate implements Weighable {
    public final static long SEP_TOKEN = -1L;
    private static final ThreadLocal<Base64Encoder> BASE64_ENCODER =
        ThreadLocal.withInitial(() -> new Base64Encoder(Base64.URL));
    private static final ThreadLocal<Base64Decoder> BASE64_DECODER =
        ThreadLocal.withInitial(() -> new Base64Decoder(Base64.URL));

    protected final long[] hashes;

    protected String attributes; // unparsed json array

    private final int weight;

    public BaseTemplate(
        final long[] hashes,
        final String attributes,
        final int weight)
    {
        this.hashes = hashes;
        this.attributes = attributes;
        this.weight = weight;
    }

    @Override
    public int weight() {
        return weight;
    }

    // Serialization
    public static String hashesToString(long[] hashes) {
        Base64Encoder encoder = BASE64_ENCODER.get();
        encoder.process(hashesToBytes(hashes));
        return encoder.toString();
    }

    public static byte[] hashesToBytes(long[] hashes) {
        ByteBuffer buf = ByteBuffer.allocate(Long.BYTES * hashes.length);
        buf.asLongBuffer().put(hashes);
        return buf.array();
    }

    public static long[] hashesFromString(String s) {
        Base64Decoder decoder = BASE64_DECODER.get();
        try {
            decoder.process(s.toCharArray());
        } catch (IOException e) {
            return null;
        }
        return hashesFromBytes(decoder.processWith(ByteBufferFactory.INSTANCE));
    }

    public static long[] hashesFromBytes(ByteBuffer data) {
        long[] res = new long[data.remaining() / Long.BYTES];
        data.asLongBuffer().get(res);
        return res;
    }

    /**
     * Longest Common Subsequence
     * TODO: weight for SEP or pathfinding to minimize SEP
     */
    public static long[] lcs(long[] x, long[] y) {
        int n = x.length;
        int m = y.length;
        int[][] dp = new int[n + 1][m + 1];
        for (int i = 1; i <= n; i++) {
            for (int j = 1; j <= m; j++) {
                if (x[i - 1] == y[j - 1]) {
                    dp[i][j] = dp[i - 1][j - 1] + 1;
                } else {
                    dp[i][j] = Math.max(dp[i - 1][j], dp[i][j - 1]);
                }
            }
        }

        int i = n;
        int j = m;
        // 40 is guesstimated amount of SEPs
        LongList common = new LongList(dp[n][m] + 40);
        while (i > 0 && j > 0) {
            if (x[i - 1] == y[j - 1]) {
                if (x[i - 1] != SEP_TOKEN || !endsWithSep(common)) {
                    common.addLong(x[i - 1]);
                }
                --i;
                --j;
            } else {
                if (!endsWithSep(common)) {
                    common.addLong(SEP_TOKEN);
                }
                if (dp[i][j] == dp[i - 1][j]) {
                    --i;
                } else {
                    --j;
                }
            }
        }
        if ((i > 0 || j > 0) && !endsWithSep(common)) {
            common.addLong(SEP_TOKEN);
        }
        long[] commonArray = common.toLongArray();
        reverse(commonArray);
        return commonArray;
    }

    private static void reverse(long[] array) {
        int end = array.length >> 1;
        for (int i = 0; i < end; ++i) {
            long temp = array[i];
            array[i] = array[array.length - i - 1];
            array[array.length - i - 1] = temp;
        }
    }

    private static boolean endsWithSep(LongList common) {
        int size = common.size();
        return size > 0 && common.getLong(size - 1) == SEP_TOKEN;
    }

    protected static int[] lcsSizes(long[] x, long[] y) {
        int n = x.length;
        int m = y.length;
        int[][] dp = new int[2][m + 1];
        for (int i = 0; i <= n; i++) {
            for (int j = 0; j <= m; j++) {
                if (i == 0 || j == 0) {
                    dp[i & 1][j] = 0;
                } else if (x[i - 1] == y[j - 1]) {
                    dp[i & 1][j] = dp[1 - i & 1][j - 1] + 1;
                } else {
                    dp[i & 1][j] =
                        Math.max(dp[1 - i & 1][j], dp[i & 1][j - 1]);
                }
            }
        }
        return dp[n & 1];
    }

    /**
     * 1.3 times slower, than lcs, but O(x.len + y.len) memory
     * TODO: less of array copying
     * returns non-empty
     * SEP on ends, if end is not matched
     * no consecutive SEPs
     * <p>
     * Now does not used
     */
    public static long[] lcsHirschberg(long[] x, long[] y) {
        if (y.length == 0) {
            return new long[]{SEP_TOKEN};
        }
        if (y.length == 1) {
            if (x.length == 1) {
                if (x[0] == y[0]) {
                    return x;
                } else {
                    return new long[]{SEP_TOKEN};
                }
            } else {
                long[] temp = x;
                x = y;
                y = temp;
            }
        }

        if (x.length == 0) {
            return new long[]{SEP_TOKEN};
        }
        if (x.length == 1) {
            if (x[0] == SEP_TOKEN) {
                return x;
            }
            if (y[0] == x[0]) {
                return new long[]{x[0], SEP_TOKEN};
            }
            if (y[y.length - 1] == x[0]) {
                return new long[]{SEP_TOKEN, x[0]};
            }

            for (long i : y) {
                if (i == x[0]) {
                    return new long[]{SEP_TOKEN, x[0], SEP_TOKEN};
                }
            }
            return new long[]{SEP_TOKEN};
        }

        int n = x.length;
        int m = y.length;
        int xDiv = (n + 1) / 2;
        long[] firstHalfX = Arrays.copyOfRange(x, 0, xDiv);
        long[] secondHalfX = Arrays.copyOfRange(x, xDiv, x.length);
        int[] prefs = lcsSizes(firstHalfX, y);
        int[] suffs = lcsSizes(reversed(secondHalfX), reversed(y));
        int matchedInFirst = -1;
        int resVal = -1;
        for (int i = 0; i <= m; ++i) {
            int curVal = prefs[i] + suffs[m - i];
            if (curVal > resVal) {
                resVal = curVal;
                matchedInFirst = i;
            }
        }

        if (resVal == 0) {
            return new long[]{SEP_TOKEN};
        }
        if (resVal == x.length) {
            return x;
        }
        long[] firstHalf = lcsHirschberg(
            firstHalfX,
            Arrays.copyOfRange(y, 0, matchedInFirst));
        long[] secondHalf = lcsHirschberg(
            secondHalfX,
            Arrays.copyOfRange(y, matchedInFirst, y.length));

        if (firstHalf[firstHalf.length - 1] == SEP_TOKEN) {
            if (secondHalf[0] == SEP_TOKEN) {
                long[] res =
                    new long[firstHalf.length + secondHalf.length - 1];
                System.arraycopy(firstHalf, 0, res, 0, firstHalf.length);
                System.arraycopy(
                    secondHalf,
                    1,
                    res,
                    firstHalf.length,
                    secondHalf.length - 1);
                return res;
            }
        }
        long[] res = new long[firstHalf.length + secondHalf.length];
        System.arraycopy(firstHalf, 0, res, 0, firstHalf.length);
        System.arraycopy(
            secondHalf, 0, res, firstHalf.length, secondHalf.length);
        return res;

    }

    private static long[] reversed(long[] orig) {
        long[] res = new long[orig.length];
        for (int i = 0; i < orig.length; ++i) {
            res[orig.length - i - 1] = orig[i];
        }
        return res;
    }

    /**
     * Uses KMP algorithm;
     * for each non-sep template's token sequence calculates prefix function:
     * [that sequence] SEP [unprocessed part of candidate]
     * until match or eof
     *
     * @return delta tokens in candidate, separated by SEP
     * SEP count is SEP count in template - 1
     */
    @Nullable
    public LongList checkMatch(final long[] candidate) {
        if (hashes.length > candidate.length) {
            return null;
        }
        int candidatePos = 0;
        int tokensPos = 0;

        while (tokensPos < hashes.length) {
            if (hashes[tokensPos] == SEP_TOKEN) {
                ++tokensPos;
                break;
            }
            if (hashes[tokensPos] != candidate[candidatePos]) {
                return null;
            }
            ++candidatePos;
            ++tokensPos;
        }

        int sequenceStart = tokensPos;
        int[] pfunc = new int[hashes.length];
        LongList delta = new LongList(candidate.length - hashes.length + 40);
        for (; tokensPos < hashes.length; ++tokensPos) {
            if (sequenceStart == tokensPos) {
                pfunc[0] = 0;
            } else {
                int k = pfunc[tokensPos - sequenceStart - 1];
                while (hashes[sequenceStart + k] != hashes[tokensPos] && k > 0)
                {
                    k = pfunc[k - 1];
                }
                if (hashes[sequenceStart + k] == hashes[tokensPos]) {
                    ++k;
                }
                pfunc[tokensPos - sequenceStart] = k;
            }

            if (hashes[tokensPos] == SEP_TOKEN
                || tokensPos + 1 == hashes.length)
            {
                int sequenceLen = tokensPos - sequenceStart + 1;
                if (hashes[tokensPos] == SEP_TOKEN) {
                    --sequenceLen;
                }
                int candidateStart = candidatePos;
                // advance in candidate to calculate:
                // pfunc for sequence
                // tokens[sequenceStart:tokensPos] + [SEP]
                //                                 + candidate[candidateStart:]
                // only previous value of pfunc required
                int prevPfunc = 0; // pfunc[SEP] = 0
                while (prevPfunc < sequenceLen) {
                    if (candidatePos == candidate.length) {
                        return null;
                    }

                    int k = prevPfunc;
                    // left one is always in tokens part, as candidate has no
                    // SEP, and
                    // k <= sequenceLen < tokens.length - sequenceStart
                    while (hashes[sequenceStart + k] != candidate[candidatePos]
                        && k > 0) {
                        k = pfunc[k - 1];
                    }
                    if (hashes[sequenceStart + k] == candidate[candidatePos]) {
                        ++k;
                    }
                    prevPfunc = k;
                    ++candidatePos;
                }
                delta.addLongs(
                    candidate,
                    candidateStart,
                    candidatePos - candidateStart - sequenceLen);
                delta.addLong(SEP_TOKEN);

                sequenceStart = tokensPos + 1;
            }
        }

        if (candidatePos < candidate.length) {
            delta.addLongs(
                candidate,
                candidatePos,
                candidate.length - candidatePos);
        } else {
            if (!delta.isEmpty()) {
                delta.removeLong(delta.size() - 1); // trailing SEP
            }
        }

        return delta;
    }


    public long[] tokens() {
        return hashes;
    }

    public String attributes() {
        return attributes;
    }
}
