package ru.yandex.mail.so.templatemaster.templates;

import ru.yandex.digest.Fnv;

/**
 * Candidate template, can be evicted
 */
public class UnstableTemplate extends BaseTemplate {
    protected final long created;
    private final long url;
    private final Digest digest;
    private int hits; // number of mails, merged into this
    private int lenSum; // total amount of tokens in merged mails

    public UnstableTemplate(
        final long[] hashes,
        final long url,
        final String attributes,
        final long created,
        final int hits,
        final int lenSum)
    {
        this(
            hashes,
            Digest.fromTokens(hashes),
            url,
            attributes,
            created,
            hits,
            lenSum);
    }

    private UnstableTemplate(
        final long[] hashes,
        final Digest digest,
        final long url,
        final String attributes,
        final long created,
        final int hits,
        final int lenSum)
    {
        super(
            hashes,
            attributes,
            36
            + 32 + digest.sortedHashes.length * Integer.BYTES
            + 24 + hashes.length * Long.BYTES);
        this.hits = hits;
        this.lenSum = lenSum;
        this.created = created;
        this.url = url;
        this.digest = digest;
    }

    public static UnstableTemplate merge(
        UnstableTemplate a,
        UnstableTemplate b,
        long[] commonTokens)
    {
        return new UnstableTemplate(
            commonTokens,
            a.url ^ b.url,
            mergeAttributes(a.attributes, b.attributes),
            Math.max(a.created, b.created),
            a.hits + b.hits,
            a.lenSum + b.lenSum);
    }

    private static String mergeAttributes(String a, String b) {
        if (a.length() < 3) {
            return b;
        } else {
            return a.substring(0, a.length() - 1) + ',' + b.substring(1);
        }
    }

    public static UnstableTemplate fromSingle(
        long[] tokens,
        long created,
        String attributes)
    {
        long url = Fnv.FNV64_INIT;
        for (long i : tokens) {
            url = url * Fnv.FNV64_PRIME ^ i;
        }
        return new UnstableTemplate(
            tokens,
            url,
            attributes,
            created,
            1,
            tokens.length);
    }

    public Digest digest() {
        return digest;
    }

    /**
     * preciseDistance <= fastDistance
     * common tokens, ignoring order
     */
    public double fastDistance(UnstableTemplate other) {
        double digestIntersection = digest.commonHashes(other.digest);
        return 1 - digestIntersection * (hits + other.hits)
            / (this.lenSum + other.lenSum);
    }

    public PreciseDistance preciseDistance(UnstableTemplate other) {
        long[] lcs = BaseTemplate.lcs(hashes, other.hashes);
        double common = 0;
        for (long i : lcs) {
            if (i != SEP_TOKEN) {
                ++common;
            }
        }
        return new PreciseDistance(
            1 - common * (hits + other.hits)
                / (this.lenSum + other.lenSum),
            lcs);
    }

    public void registerHit(UnstableTemplate other) {
        ++this.hits;
        this.lenSum += other.hashes.length;
        this.attributes = mergeAttributes(this.attributes, other.attributes);
    }

    public long created() {
        return created;
    }

    public long url() {
        return url;
    }

    public int hits() {
        return hits;
    }

    public int lenSum() {
        return lenSum;
    }

    public static class PreciseDistance implements Comparable<PreciseDistance>
    {
        private final double distance;
        private final long[] commonTokens;

        public PreciseDistance(double distance, long[] commonTokens) {
            this.distance = distance;
            this.commonTokens = commonTokens;
        }

        public double distance() {
            return distance;
        }

        public long[] commonTokens() {
            return commonTokens;
        }

        @Override
        public int compareTo(PreciseDistance other) {
            return Double.compare(distance, other.distance);
        }
    }
}
