#pragma once
#include <mail/template_master/lib/types/context.h>
#include <mail/template_master/lib/db/db_info.h>
#include <mail/template_master/lib/utils/utils.h>
#include <mail/template_master/lib/utils/errors.h>
#include <mail/template_master/lib/db/shard_resolver.h>
#include <mail/template_master/lib/router/router_module.h>
#include <mail/template_master/lib/types/expected.h>
#include <mail/template_master/lib/db/source_map.h>

#include <boost/system/error_code.hpp>
#include <boost/range/adaptors.hpp>

#include <util/random/random.h>

#include <vector>

namespace NTemplateMaster::NDatabase {

enum class EExecutionPolicy {
    Master,
    ReplicaOrMaster
};

template<typename TResolverPtr>
class TConnectionProvider {
public:
    TConnectionProvider(TSourceMapPtr pool, TResolverPtr shardResolver, TConfig config)
        : SourceMap(std::move(pool))
        , Resolver(std::move(shardResolver))
        , Config(std::move(config))
    {}

    decltype(auto) GetConnectionProvider(TDatabaseInfo dbInfo, TIOContext* io) {
        return (*SourceMap)(dbInfo.MakeConnInfo())[*io];
    }

    auto GetDatabaseInfo(NTemplateMaster::TContextPtr context, EExecutionPolicy policy, TYield yield) {
        using namespace NTemplateMaster::NErrors;

        boost::system::error_code ec;
        auto shards = Resolver->GetAllShards(context, yield[ec]);
        if (ec) {
            LOGDOG_(context->GetLogger(), error, NTemplateMaster::NLog::error_code=ec)
            return yamail::make_expected_from_error<TDatabaseInfo>(ec);
        }
        auto&& [shardId, shard] = *(shards.begin());

        auto databases = shard.databases;
        ::sharpei::client::Shard::Database result;
        std::vector<::sharpei::client::Shard::Database> aliveDbs;

        boost::copy(databases | boost::adaptors::filtered([](auto&& db) {
            return db.status == "alive";
        }), std::back_inserter(aliveDbs));

        if (!aliveDbs.empty()) {
            if (policy == EExecutionPolicy::Master) {
                for (auto&& db : aliveDbs) {
                    if (db.role == "master") {
                        return MakeExpected<TDatabaseInfo>({db, Config.User});
                    }
                }
            } else if (policy == EExecutionPolicy::ReplicaOrMaster) {
                return MakeExpected<TDatabaseInfo>(
                        {aliveDbs[RandomNumber<size_t>() % aliveDbs.size()], Config.User});
            }
        }
        ec = MakeErrorCode(EError::RequestedDbNotFound);
        LOGDOG_(context->GetLogger(), error, NTemplateMaster::NLog::error_code=ec)
        return yamail::make_expected_from_error<TDatabaseInfo>(ec);
    }

private:
    TSourceMapPtr SourceMap;
    TResolverPtr Resolver;
    TConfig Config;
};

using TConnectionProviderPtr = std::shared_ptr<TConnectionProvider<TShardResolverPtr>>;
using TConnectionProviderType = decltype(std::declval<TConnectionProvider<TShardResolverPtr>>().GetConnectionProvider(
        std::declval<TDatabaseInfo>(), std::declval<TIOContext*>()));

inline auto GetConnectionProvider(
        NTemplateMaster::TContextPtr context,
        TConnectionProviderPtr connProvider,
        EExecutionPolicy policy,
        TIOContext* io,
        TYield yield)
{
    auto dbInfo = connProvider->GetDatabaseInfo(context, policy, yield);
    if (!dbInfo) {
        return yamail::make_expected_from_error<TConnectionProviderType>(dbInfo.error());
    }
    return MakeExpected<TConnectionProviderType>(connProvider->GetConnectionProvider(dbInfo.value(), io));
}
}
