#pragma once
#include <mail/template_master/lib/types/context.h>
#include <mail/template_master/lib/utils/utils.h>
#include <mail/template_master/lib/unistat/unistat.h>
#include <mail/template_master/lib/db/database_module.h>
#include <mail/template_master/lib/db/config.h>
#include <mail/template_master/lib/db/connection_provider.h>
#include <mail/template_master/lib/db/shard_resolver.h>
#include <mail/template_master/lib/db/source_map.h>
#include <mail/template_master/lib/db/operations/operation_traits.h>
#include <mail/template_master/lib/db/operations/find_similar_templates.h>
#include <mail/template_master/lib/db/operations/save_template.h>
#include <mail/template_master/lib/db/operations/find_template_by_stable_sign.h>
#include <mail/template_master/lib/db/operations/delete_template.h>
#include <mail/template_master/lib/db/operations/merge_templates.h>
#include <mail/template_master/lib/db/operations/index_template_features.h>
#include <mail/template_master/lib/db/operations/remove_templates_from_features_index.h>
#include <mail/template_master/lib/db/operations/get_ready_tasks.h>
#include <mail/template_master/lib/db/operations/finish_task.h>
#include <mail/template_master/lib/db/operations/index_feature.h>
#include <mail/template_master/lib/db/operations/move_timeout_task.h>
#include <mail/template_master/lib/db/operations/update_search_flag.h>
#include <mail/template_master/lib/db/operations/get_unused_features.h>
#include <mail/template_master/lib/db/queries/query_repository.h>

#include <mail/yplatform/include/yplatform/module.h>
#include <mail/yplatform/include/yplatform/reactor.h>
#include <mail/yreflection/include/yamail/data/deserialization/ptree_reader.h>

#include <boost/asio/io_service.hpp>
#include <boost/asio/spawn.hpp>

namespace NTemplateMaster::NDatabase {

class TDatabaseImpl: public IDatabase {
public:
    TDatabaseImpl(yplatform::reactor& reactor)
        : Reactor(reactor)
    {}

    void init(const yplatform::ptree& config) {
        Config = yamail::data::deserialization::fromPtree<TConfig>(config.get_child("pg"));
        QueryRepository = NQuery::MakeQueryRepository(Config.QueryConf);
        SourceMap = std::make_shared<TSourceMap>(TConnectionSourceCtor({}, {}, Config.ConnectionPoolConfig));
        auto shardResolver = std::make_shared<TShardResolver>(Config.SharpeiConfig);
        ConnectionProvider = std::make_shared<TConnectionProvider<TShardResolverPtr>>(SourceMap, shardResolver, Config);
    }

    TExpected<TDatabaseTemplates> FindSimilarTemplates(
            NTemplateMaster::TContextPtr context,
            const TTemplateFeaturesSet& hashes,
            int32_t limit,
            int32_t featuresLimit,
            TYield yield) noexcept override {
        TAsyncCompletion<void(TExpected<TDatabaseTemplates>)> init(yield);
        auto handler = init.completion_handler;

        auto io = GetIo();
        Operations::TFindSimilarTemplatesOp op(QueryRepository, hashes, limit, featuresLimit);
        boost::asio::spawn(*io, NTemplateMaster::NUnistat::WrapWithLog(
                [io, context, op, handler, self=SharedFromThis()](TYield yield) mutable {
            const auto result = (*self)(context, yield, op, EExecutionPolicy::ReplicaOrMaster, io);
            NTemplateMaster::NUnistat::CollectReturnResult(result, "db_FindSimilarTemplates");
            return NUtils::Dispatch(handler, result);
        }, context, "application_db_FindSimilarTemplates"), NUtils::kCoroutineAttributes);
        return init.result.get();
    }

    TExpected<TOptional<TDatabaseTemplatePtr>> FindTemplateByStableSign(
            NTemplateMaster::TContextPtr context,
            TTemplateStableSign stableSign,
            TYield yield) noexcept override {
        TAsyncCompletion<void(TExpected<TOptional<TDatabaseTemplatePtr>>)> init(yield);
        auto handler = init.completion_handler;

        auto io = GetIo();
        Operations::TFindTemplateByStableSignOp op(QueryRepository, stableSign);
        boost::asio::spawn(*io, NTemplateMaster::NUnistat::WrapWithLog(
                [io, op, context, handler, self=SharedFromThis()](TYield yield) mutable {
            const auto result = (*self)(context, yield, op, EExecutionPolicy::ReplicaOrMaster, io);
            NTemplateMaster::NUnistat::CollectReturnResult(result, "db_FindTemplateByStableSign");
            return NUtils::Dispatch(handler, result);
        }, context, "application_db_FindTemplateByStableSign"), NUtils::kCoroutineAttributes);
        return init.result.get();
    }

    TExpected<void> SaveTemplate(
            NTemplateMaster::TContextPtr context,
            TDatabaseTemplatePtr templ,
            int32_t featuresLimit,
            TTemplatesStableSigns parentIds,
            TYield yield) noexcept override {
        TAsyncCompletion<void(TExpected<void>)> init(yield);
        auto handler = init.completion_handler;

        auto io = GetIo();
        Operations::TSaveTemplateOp op(QueryRepository, templ, featuresLimit, parentIds);
        boost::asio::spawn(*io, NTemplateMaster::NUnistat::WrapWithLog(
                [io, context, op, handler, self=SharedFromThis()](TYield yield) mutable {
            const auto result = (*self)(context, yield, op, EExecutionPolicy::Master, io);
            NTemplateMaster::NUnistat::CollectReturnResult(result, "db_SaveTemplate");
            return NUtils::Dispatch(handler, result);
        }, context, "application_db_SaveTemplate"), NUtils::kCoroutineAttributes);
        return init.result.get();
    }

    TExpected<TTemplateFeaturesSet> GetUnusedFeatures(
            NTemplateMaster::TContextPtr context,
            TYield yield) noexcept override
    {
        TAsyncCompletion<void(TExpected<TTemplateFeaturesSet>)> init(yield);
        auto handler = init.completion_handler;

        auto io = GetIo();
        Operations::TGetUnusedFeaturesOp op(QueryRepository);
        boost::asio::spawn(*io, NTemplateMaster::NUnistat::WrapWithLog(
                [io, op, context, handler, self=SharedFromThis()](TYield yield) mutable {
                    const auto result = (*self)(context, yield, op, EExecutionPolicy::Master, io);
                    NTemplateMaster::NUnistat::CollectReturnResult(result, "db_GetUnusedFeatures");
                    return NUtils::Dispatch(handler, result);
                }, context, "application_db_GetUnusedFeatures"), NUtils::kCoroutineAttributes);
        return init.result.get();
    }

    TExpected<size_t> DeleteTemplate(
            NTemplateMaster::TContextPtr context,
            TTemplateStableSign stableSign,
            TYield yield) noexcept override {
        TAsyncCompletion<void(TExpected<size_t>)> init(yield);
        auto handler = init.completion_handler;

        auto io = GetIo();
        Operations::TDeleteTemplateOp op(QueryRepository, stableSign);
        boost::asio::spawn(*io, NTemplateMaster::NUnistat::WrapWithLog(
                [io, op, context, handler, self=SharedFromThis()](TYield yield) mutable {
            const auto result = (*self)(context, yield, op, EExecutionPolicy::Master, io);
            NTemplateMaster::NUnistat::CollectReturnResult(result, "db_DeleteTemplate");
            return NUtils::Dispatch(handler, result);
        }, context, "application_db_DeleteTemplate"), NUtils::kCoroutineAttributes);
        return init.result.get();
    }

    TExpected<void> UpdateSearchFlag(
            NTemplateMaster::TContextPtr context,
            bool search,
            TTemplatesStableSigns stableSigns,
            TYield yield) noexcept override {
        TAsyncCompletion<void(TExpected<void>)> init(yield);
        auto handler = init.completion_handler;

        auto io = GetIo();
        Operations::TUpdateSearchFlagOp op(QueryRepository, search, stableSigns);
        boost::asio::spawn(*io, NTemplateMaster::NUnistat::WrapWithLog(
                [io, op, context, handler, self=SharedFromThis()](TYield yield) mutable {
            const auto result = (*self)(context, yield, op, EExecutionPolicy::Master, io);
            NTemplateMaster::NUnistat::CollectReturnResult(result, "db_UpdateSearchFlag");
            return NUtils::Dispatch(handler, result);
        }, context, "application_db_UpdateSearchFlag"), NUtils::kCoroutineAttributes);
        return init.result.get();
    }

    TExpected<void> IndexTemplateFeatures(
            NTemplateMaster::TContextPtr context,
            TTemplateFeaturesSet features,
            TTemplateStableSign stableSign,
            TYield yield) noexcept override {
        TAsyncCompletion<void(TExpected<void>)> init(yield);
        auto handler = init.completion_handler;

        auto io = GetIo();
        Operations::TIndexTemplateFeaturesOp op(QueryRepository, stableSign, features);
        boost::asio::spawn(*io, NTemplateMaster::NUnistat::WrapWithLog(
                [io, op, context, handler, self=SharedFromThis()](TYield yield) mutable {
            const auto result = (*self)(context, yield, op, EExecutionPolicy::Master, io);
            NTemplateMaster::NUnistat::CollectReturnResult(result, "db_IndexTemplateFeatures");
            return NUtils::Dispatch(handler, result);
        }, context, "application_db_IndexTemplateFeatures"), NUtils::kCoroutineAttributes);
        return init.result.get();
    }

    TExpected<void> IndexFeature(
            NTemplateMaster::TContextPtr context,
            TTemplateFeature feature,
            TYield yield) noexcept override {
        TAsyncCompletion<void(TExpected<void>)> init(yield);
        auto handler = init.completion_handler;

        auto io = GetIo();
        Operations::TIndexFeatureOp op(QueryRepository, feature);
        boost::asio::spawn(*io, NTemplateMaster::NUnistat::WrapWithLog(
                [io, op, context, handler, self=SharedFromThis()](TYield yield) mutable {
            const auto result = (*self)(context, yield, op, EExecutionPolicy::Master, io);
            NTemplateMaster::NUnistat::CollectReturnResult(result, "db_IndexFeature");
            return NUtils::Dispatch(handler, result);
        }, context, "application_db_IndexFeature"), NUtils::kCoroutineAttributes);
        return init.result.get();
    }

    TExpected<void> RemoveTemplateFromFeaturesIndex(
            NTemplateMaster::TContextPtr context,
            TTemplateStableSign stableSign,
            TYield yield) noexcept override {
        TAsyncCompletion<void(TExpected<void>)> init(yield);
        auto handler = init.completion_handler;

        auto io = GetIo();
        Operations::TRemoveTemplatesFromFeaturesIndexOp op(QueryRepository, stableSign);
        boost::asio::spawn(*io, NTemplateMaster::NUnistat::WrapWithLog(
                [io, op, context, handler, self=SharedFromThis()](TYield yield) mutable {
            const auto result = (*self)(context, yield, op, EExecutionPolicy::Master, io);
            NTemplateMaster::NUnistat::CollectReturnResult(result, "db_RemoveTemplateFromFeaturesIndex");
            return NUtils::Dispatch(handler, result);
        }, context, "application_db_RemoveTemplateFromFeaturesIndex"), NUtils::kCoroutineAttributes);
        return init.result.get();
    }

    TExpected<void> MoveTimeoutTask(
            NTemplateMaster::TContextPtr context,
            TYield yield) override {
        TAsyncCompletion<void(TExpected<void>)> init(yield);
        auto handler = init.completion_handler;

        auto io = GetIo();
        Operations::TMoveTimeoutTaskOp op(QueryRepository);
        boost::asio::spawn(*io, NTemplateMaster::NUnistat::WrapWithLog(
                [io, op, context, handler, self=SharedFromThis()](TYield yield) mutable {
            const auto result = (*self)(context, yield, op, EExecutionPolicy::Master, io);
            NTemplateMaster::NUnistat::CollectReturnResult(result, "db_MoveTimeoutTask");
            return NUtils::Dispatch(handler, result);
        }, context, "application_db_MoveTimeoutTask"), NUtils::kCoroutineAttributes);
        return init.result.get();
    }

    TExpected<TTasksInfos> GetReadyTasks(
            NTemplateMaster::TContextPtr context,
            int32_t limit,
            TYield yield) noexcept override {
        TAsyncCompletion<void(TExpected<TTasksInfos>)> init(yield);
        auto handler = init.completion_handler;

        auto io = GetIo();
        Operations::TGetReadyTasksOp op(QueryRepository, limit);
        boost::asio::spawn(*io, NTemplateMaster::NUnistat::WrapWithLog(
                [io, op, context, handler, self=SharedFromThis()](TYield yield) mutable {
            const auto result = (*self)(context, yield, op, EExecutionPolicy::Master, io);
            NTemplateMaster::NUnistat::CollectReturnResult(result, "db_GetReadyTasks");
            return NUtils::Dispatch(handler, result);
        }, context, "application_db_GetReadyTasks"), NUtils::kCoroutineAttributes);
        return init.result.get();
    }

    TExpected<void> FinishTask(
            NTemplateMaster::TContextPtr context,
            std::string taskId,
            ETaskStatus taskStatus,
            TOptional<std::string> error,
            TYield yield) noexcept override {
        TAsyncCompletion<void(TExpected<void>)> init(yield);
        auto handler = init.completion_handler;

        auto io = GetIo();
        Operations::TFinishTaskOp op(QueryRepository, taskId, taskStatus, error);
        boost::asio::spawn(*io, NTemplateMaster::NUnistat::WrapWithLog(
                [io, op, context, handler, self=SharedFromThis()](TYield yield) mutable {
            const auto result = (*self)(context, yield, op, EExecutionPolicy::Master, io);
            NTemplateMaster::NUnistat::CollectReturnResult(result, "db_FinishTask");
            return NUtils::Dispatch(handler, result);
        }, context, "application_db_FinishTask"), NUtils::kCoroutineAttributes);
        return init.result.get();
    }

    void AsyncSaveTemplate(
            NTemplateMaster::TContextPtr context,
            TDatabaseTemplatePtr templ,
            int32_t featuresLimit,
            TTemplatesStableSigns parentIds,
            std::function<void(TExpected<void>)> callback) override {
        auto io = GetIo();
        Operations::TSaveTemplateOp op(QueryRepository, templ, featuresLimit, parentIds);
        boost::asio::spawn(*io, NTemplateMaster::NUnistat::WrapWithLog(
                [io, callback, context, op, self=SharedFromThis()](TYield yield) mutable {
            const auto result = (*self)(context, yield, op, EExecutionPolicy::Master, io);
            NTemplateMaster::NUnistat::CollectReturnResult(result, "db_AsyncSaveTemplate");
            callback(result);
        }, context, "application_db_AsyncSaveTemplate"), NUtils::kCoroutineAttributes);
    }

    void AsyncMergeTemplates(
            NTemplateMaster::TContextPtr context,
            TDatabaseTemplatePtr templ,
            int32_t featuresLimit,
            TTemplatesStableSigns stableSigns,
            std::function<void(TExpected<bool>)> callback) override {
        auto io = GetIo();
        Operations::TMergeTemplatesOp op(QueryRepository, templ, featuresLimit, stableSigns);
        boost::asio::spawn(*io, NTemplateMaster::NUnistat::WrapWithLog(
                [io, callback, context, op, self=SharedFromThis()](TYield yield) mutable {
            const auto result = (*self)(context, yield, op, EExecutionPolicy::Master, io);
            NTemplateMaster::NUnistat::CollectReturnResult(result, "db_AsyncMergeTemplates");
            callback(result);
        }, context, "application_db_AsyncMergeTemplates"), NUtils::kCoroutineAttributes);
    }

private:
    template<typename TOperation>
    Operations::TOperationReturnType<TOperation, TConnectionProviderType>
    operator()(
            NTemplateMaster::TContextPtr context,
            TYield yield,
            TOperation&& op,
            EExecutionPolicy policy,
            TIOContext* io)
    {
        auto provider = GetConnectionProvider(context, ConnectionProvider, policy, io, yield);
        if (!provider) {
            return yamail::make_unexpected(provider.error());
        }
        return op(context, provider.value(), Config.RequestTimeout, yield);
    }

    std::shared_ptr<TDatabaseImpl> SharedFromThis() {
        return std::dynamic_pointer_cast<TDatabaseImpl>(shared_from_this());
    }

    TIOContext* GetIo() const {
        return Reactor.io();
    }

    TConfig Config;
    NQuery::TQueryRepository QueryRepository;
    TSourceMapPtr SourceMap;
    TConnectionProviderPtr ConnectionProvider;
    yplatform::reactor& Reactor;
};

}
