#pragma once
#include <mail/template_master/lib/db/queries/query_repository.h>
#include <mail/template_master/lib/types/context.h>
#include <mail/template_master/lib/utils/utils.h>
#include <mail/template_master/lib/types/expected.h>
#include <mail/template_master/lib/db/operations/operation_traits.h>

#include <ozo/result.h>
#include <ozo/request.h>
#include <ozo/execute.h>
#include <ozo/transaction.h>
#include <ozo/shortcuts.h>
#include <ozo/pg/types/jsonb.h>

#include <vector>

namespace NTemplateMaster::NDatabase::Operations {

template<typename TQueryRepository>
class TFindSimilarTemplatesOp {
private:
    using TFindSimilarTemplates = NTemplateMaster::NDatabase::NQuery::TFindSimilarTemplates;
    using TSetSimilarThreshold = NTemplateMaster::NDatabase::NQuery::TSetSimilarThreshold;
    using TTemplateBody = ozo::pg::jsonb;
public:
    TFindSimilarTemplatesOp(
            TQueryRepository queryRepository,
            const ::NTemplateMaster::TTemplateFeaturesSet& hashes,
            int_fast32_t limit,
            int_fast32_t featuresLimit)
        : QueryRepository(std::move(queryRepository))
        , Hashes(std::move(hashes))
        , Limit(limit)
        , FeaturesLimit(featuresLimit)
    {}

    template<typename TProvider>
    TExpected<TDatabaseTemplates> operator()(
            NTemplateMaster::TContextPtr context,
            TProvider&& provider,
            ozo::time_traits::duration requestTimeout,
            TYield yield) {
        using Adl::request;
        using Adl::execute;
        using Adl::begin;
        using Adl::commit;

        boost::system::error_code ec;
        std::vector<typename TFindSimilarTemplates::result_type> result;
        TFindSimilarTemplates findSimilarTemplatesParams;
        findSimilarTemplatesParams.sign = {Hashes.begin(), Hashes.end()};
        findSimilarTemplatesParams.limit = Limit;
        findSimilarTemplatesParams.features_limit = FeaturesLimit;
        const auto searchQ = QueryRepository.template make_query<TFindSimilarTemplates>(findSimilarTemplatesParams);
        const auto setQ = QueryRepository.template make_query<TSetSimilarThreshold>();

        auto transaction = begin(std::forward<TProvider>(provider), requestTimeout, yield[ec]);
        if (ec) {
            LogOzoError(context, ec, "begin", transaction);
            return yamail::make_unexpected(ec);
        }
        execute(transaction, setQ, requestTimeout, yield[ec]);
        if (ec) {
            LogOzoError(context, ec, setQ, transaction);
            return yamail::make_unexpected(ec);
        }
        request(transaction, searchQ, requestTimeout, ozo::into(result), yield[ec]);
        if (ec) {
            LogOzoError(context, ec, searchQ, transaction);
            return yamail::make_unexpected(ec);
        }
        commit(std::move(transaction), requestTimeout, yield[ec]);
        if (ec) {
            LogOzoError(context, ec, "commit", transaction);
            return yamail::make_unexpected(ec);
        }
        return ConvertDbAnswerToTemplates(result);
    }
private:
    template<typename TRowsOfTemplates>
    static TDatabaseTemplates ConvertDbAnswerToTemplates(const TRowsOfTemplates& rows) {
        TDatabaseTemplates result;
        for (auto&& row : rows) {
            const auto& [stableSign, sign, json, _] = row;
            result.emplace_back(std::make_shared<TDatabaseTemplate>(
                stableSign,
                TTemplateFeaturesSet(sign.begin(), sign.end()),
                json.raw_string(),
                TJsonAttributesArray{}));
        }
        return result;
    }
private:
    TQueryRepository QueryRepository;
    const ::NTemplateMaster::TTemplateFeaturesSet Hashes;
    const int_fast32_t Limit;
    const int_fast32_t FeaturesLimit;
};

}
